/*     
 **********************************************************************
 *     cardmo.c - MIDI UART output HAL for emu10k1 driver 
 *     Copyright 1999, 2000 Creative Labs, Inc. 
 * 
 ********************************************************************** 
 * 
 *     Date                 Author          Summary of changes 
 *     ----                 ------          ------------------ 
 *     October 20, 1999     Bertrand Lee    base code release 
 *     November 2, 1999     Alan Cox        cleaned up
 * 
 ********************************************************************** 
 * 
 *     This program is free software; you can redistribute it and/or 
 *     modify it under the terms of the GNU General Public License as 
 *     published by the Free Software Foundation; either version 2 of 
 *     the License, or (at your option) any later version. 
 * 
 *     This program is distributed in the hope that it will be useful, 
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *     GNU General Public License for more details. 
 * 
 *     You should have received a copy of the GNU General Public 
 *     License along with this program; if not, write to the Free 
 *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, 
 *     USA. 
 * 
 ********************************************************************** 
 */

#include "hwaccess.h"
#include "halstr.h"
#include "cardmo.h"


/*
 *   Internal Function Declaration 
 */

static int sblive_mpuoutIrqCallback(unsigned long event, unsigned long refdata,
				 unsigned long param);

static int sblive_mpuoutDpcCallback(unsigned long refdata, unsigned long param1,
				 unsigned long param2);


//*****************************************************************************
//*int    sblive_mpuoutInit(PICARDMIDI pICardMidi, PICARD pICard )          *
//*                                                                           *
//*Function:   Initialize the parameters + reset the MPU port                 *
//*****************************************************************************
int sblive_mpuoutInit(struct sblive_mpuout *card_mpuout, struct sblive_hw *sb_hw)
{
	u8 name[128];
	DPF("sblive_mpuoutInit");
	ASSERT(card_mpuout);
	TESTMAGIC(sb_hw);

	memset(card_mpuout, 0, sizeof(struct sblive_mpuout));

	card_mpuout->intr = TRUE;
	card_mpuout->status = FLAGS_AVAILABLE;
	card_mpuout->state = CARDMIDIOUT_STATE_DEFAULT;
	card_mpuout->caps.cbsize = sizeof(struct midi_caps);
	card_mpuout->caps.support = MIDICAPS_OUTPUT;
	card_mpuout->caps.technology = MT_MIDIPORT;
	card_mpuout->caps.product = MM_CREATIVE_MIDIOUT;
	card_mpuout->caps.manufacturer = MM_CREATIVE;
	card_mpuout->caps.voices = 0;
	card_mpuout->caps.notes = 0;
	card_mpuout->caps.channelmask = 0xffff;
	card_mpuout->caps.caps = CARDMIDI_OUT;

	card_mpuout->dpc.is_active = FALSE;
	card_mpuout->dpc.refdata = (u32) sb_hw;
	card_mpuout->dpc.DPCCallBackFn = sblive_mpuoutDpcCallback;

	spin_lock_init(&card_mpuout->lock);

	/*
	   ** Set Up the Irq Handler 
	 */
	if (sblive_irqmgrInstallIrqHandler(sb_hw,
				    (u16) (IRQTYPE_MPUOUT),
	 sblive_mpuoutIrqCallback, (u32) sb_hw) != CTSTATUS_SUCCESS) {
		DPF("CARDMIDI : IRQ in use");
		return CTSTATUS_ALLOCATED;
	} else {
		DPF("CARDMIDI : IRQ installed");
	}

	strcpy(name, IDS_EMU_MIDIOUT_PNAME);

	// Fill CardCaps
	sprintf(card_mpuout->caps.MIDIname, "%s [%lx]",
	    name, sb_hw->hwaddr);

	// Reset the MPU port
	if (hwmpuReset(sb_hw) != CTSTATUS_SUCCESS) {
		DPF("MPU hardware reset failure");
		return CTSTATUS_NOTENABLED;
	}
	SETMAGIC(card_mpuout);

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//*int    sblive_mpuoutExit(struct sblive_mpuout *card_mpuout)                    *
//*                                                                           *
//*Function:   Disable the IRQ TX and uninstall the IRQ handler               *
//*****************************************************************************

int sblive_mpuoutExit(struct sblive_hw *sb_hw)
{
	struct sblive_mpuout *card_mpuout = sb_hw->card_mpuout;
	DPF("sblive_mpuoutExit");
	TESTMAGIC(card_mpuout);
	CLEARMAGIC(card_mpuout);

	// Disable TX interrupt
	sblive_irqmgrDisableIrq(sb_hw, ENB_TX);

	// Uninstall IRQ handler
	sblive_irqmgrUninstallIrqHandler(sb_hw, IRQTYPE_MPUOUT);

	card_mpuout->status |= FLAGS_AVAILABLE;
	card_mpuout->state = CARDMIDIOUT_STATE_DEFAULT;

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//* int    sblive_mpuoutSetPowerState(struct sblive_mpuout *card_mpuout,          *
//*                                     u32     powerstate)             *
//*                                                                           *
//* Function:   Handle power management messages                              *
//*****************************************************************************
int sblive_mpuoutSetPowerState(struct sblive_hw *sb_hw, u32 powerstate)
{
	struct sblive_mpuout *card_mpuout = sb_hw->card_mpuout;
	TESTMAGIC(card_mpuout);

	switch (powerstate) {
	case POWERSTATE_STANDBY:
	case POWERSTATE_SUSPEND:
		if (!(card_mpuout->status & FLAGS_AVAILABLE))
			sblive_mpuoutReset(sb_hw, (u32) card_mpuout);
		card_mpuout->state = CARDMIDIOUT_STATE_SUSPEND;
		break;

	case POWERSTATE_STANDBYRESUME:
	case POWERSTATE_SUSPENDRESUME:
		card_mpuout->state = CARDMIDIOUT_STATE_DEFAULT;
		break;

	default:
		ASSERT(FALSE);
		return CTSTATUS_ERROR;
	}
	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//* int    sblive_mpuoutGetCaps(PICARDMIDI pICardMidi,                      *
//*                               struct midi_caps *midi_caps)                  *
//*                                                                           *
//* Function:   Returns the MPU out capabilities                              *
//*****************************************************************************
int sblive_mpuoutGetCaps(struct sblive_mpuout *card_mpuout, struct midi_caps *midi_caps)
{
	u32 cbsize;

	DPF("sblive_mpuoutGetCaps");
	TESTMAGIC(card_mpuout);

	cbsize = min(midi_caps->cbsize, card_mpuout->caps.cbsize);
	memcpy(midi_caps, &card_mpuout->caps, cbsize);
	midi_caps->cbsize = cbsize;

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//* int    sblive_mpuoutOpen(PICARDMIDI pICardMidi,                         *
//*                            struct midi_openinfo *openinfo,                 *
//*                            u32 *handle)                        *
//*                                                                           *
//* Function:   Installs the IRQ handler for the MPU out port                 *
//*             and initialize parameters                                     *
//*****************************************************************************
int sblive_mpuoutOpen(
			  struct sblive_hw *sb_hw,
			  struct midi_openinfo *openinfo,
			  u32 *handle
)
{
	struct sblive_mpuout *card_mpuout = sb_hw->card_mpuout; 
	DPF("sblive_mpuoutOpen");
	TESTMAGIC(card_mpuout);

	if (!(card_mpuout->status & FLAGS_AVAILABLE))
		return CTSTATUS_INUSE;

	/*
	   ** Copy Open Info and Mark Channel as InUse
	 */
	card_mpuout->intr = TRUE;
	card_mpuout->openinfo = *openinfo;
	card_mpuout->status &= ~FLAGS_AVAILABLE;
	card_mpuout->laststatus = 0x80;
	card_mpuout->firstmidiq = NULL;
	card_mpuout->lastmidiq = NULL;

	hwmpuReset(sb_hw);
	hwmpuAcquire(sb_hw);

	*handle = (u32) card_mpuout;

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//* int    sblive_mpuoutClose(PICARDMIDI pICardMidi,                        *
//*                             u32 handle)                         *
//*                                                                           *
//* Function:   If local buffer is not empty,                                 *
//*             return CTSTATUS_STILLPLAYING                                  *
//*             Otherwise, disable and uninstall TX IRQ                       *
//*****************************************************************************
int sblive_mpuoutClose(struct sblive_hw *sb_hw, u32 handle)
{
	struct sblive_mpuout *card_mpuout = sb_hw->card_mpuout;
	DPF("sblive_mpuoutClose");
	TESTMAGIC(card_mpuout);

	if (card_mpuout->firstmidiq) {
		DPF("Cannot close. Buffers not empty.");
		return CTSTATUS_STILLPLAYING;
	}
	// Disable TX interrupt
	sblive_irqmgrDisableIrq(sb_hw, ENB_TX);
	DPF("Disable Tx");

	hwmpuRelease(sb_hw);

	card_mpuout->status |= FLAGS_AVAILABLE;

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//* int    sblive_mpuoutAddBuffer(PICARDMIDI pICardMidi,                    *
//*                                 u32 handle,                     *
//*                                 struct midi_hdr *INFO midihdrInfo)          *
//*                                                                           *
//* Function:   If there isn't enough buffer space, reject Midi Buffer.       *
//*             Otherwise, disable TX, create object to hold Midi             *
//*             Buffer, update buffer flags and other parameters              *
//*             before enabling TX again.                                     *
//*****************************************************************************
int sblive_mpuoutAddBuffer(
			       struct sblive_hw *sb_hw,
			       u32 handle,
			       struct midi_hdr *midihdr
)
{
	struct sblive_mpuout *card_mpuout = sb_hw->card_mpuout;
	struct midi_queue *midiq;
	unsigned long flags;

	DPF("sblive_mpuoutAddBuffer");
	TESTMAGIC(card_mpuout);

	if (card_mpuout->state == CARDMIDIOUT_STATE_SUSPEND)
		return CTSTATUS_SUCCESS;

	// Update Midi Buffer flags
	midihdr->flags |= MIDIBUF_INQUEUE;		// set

	midihdr->flags &= ~MIDIBUF_DONE;	// clear

	midiq = (struct midi_queue *) kmalloc(sizeof(struct midi_queue), GFP_KERNEL);
	if (midiq == NULL) {
		// message lost
		return CTSTATUS_NOMEMORY;
	}
	midiq->next = NULL;
	midiq->qtype = 1;
	midiq->length = midihdr->bufferlength;
	midiq->sizeLeft = midihdr->bufferlength;
	midiq->midibyte = midihdr->lpData;

	midiq->refdata = (u32) midihdr;

	spin_lock_irqsave(&card_mpuout->lock, flags);

	if (card_mpuout->firstmidiq == NULL) {
		card_mpuout->firstmidiq = midiq;
		card_mpuout->lastmidiq = midiq;
	} else {
		(card_mpuout->lastmidiq)->next = midiq;
		card_mpuout->lastmidiq = midiq;
	}

	card_mpuout->intr = FALSE;
	sblive_irqmgrEnableIrq(sb_hw, ENB_TX);

	spin_unlock_irqrestore(&card_mpuout->lock, flags);

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//* int    sblive_mpuoutStart(PICARDMIDI pICardMidi,                        *
//*                             u32 handle,                         *
//*                                                                           *
//* Function:   Not Used                                                      *
//*****************************************************************************
int sblive_mpuoutStart(struct sblive_mpuout *card_mpuout, u32 handle)
{
	DPF("sblive_mpuoutStart");
	return CTSTATUS_NOTSUPPORTED;
}


//*****************************************************************************
//* int    sblive_mpuoutStop(PICARDMIDI pICardMidi,                         *
//*                            u32 handle,                          *
//*                                                                           *
//* Function:   Not Used                                                      *
//*****************************************************************************
int sblive_mpuoutStop(struct sblive_mpuout *card_mpuout, u32 handle)
{
	DPF("sblive_mpuoutStop");
	return CTSTATUS_NOTSUPPORTED;
}


//*****************************************************************************
//* int    sblive_mpuoutWriteShortData(PICARDMIDI pICardMidi,               *
//*                                      u32 handle,                *
//*                                      u32 midimsg)                   *
//*                                                                           *
//* Function:   Adds the short message to our local buffer                    *
//* Note:       This writeout must succeed at all cost ... else UART output   *
//*             will sound wierd at a sound module.                           *
//*****************************************************************************
int sblive_mpuoutWriteShortData(
				    struct sblive_hw *sb_hw,
				    u32 handle,
				    u32 midimsg
)
{
	struct sblive_mpuout *card_mpuout = sb_hw->card_mpuout;
	int msglen;
	struct midi_queue *midiq;
	int status;
	unsigned long flags;

	TESTMAGIC(card_mpuout);

	if (card_mpuout->state == CARDMIDIOUT_STATE_SUSPEND)
		return CTSTATUS_SUCCESS;

	if (midimsg & 0x80) {
		if (((u8) midimsg) < 0xF0) {
			card_mpuout->laststatus = (u8) midimsg;
			msglen = gabMsgLenChannel[((midimsg & 0xF0) - 0x80) >> 4];
		} else
			msglen = gabMsgLenSystem[midimsg & 0x0F];
	} else {
		if (card_mpuout->laststatus)
			msglen = gabMsgLenChannel[((card_mpuout->laststatus & 0xF0) - 0x80) >> 4];
		else {
			DPD("sblive_mpuoutWriteShortData error!!: midimsg = ", midimsg);
			return CTSTATUS_ERROR;
		}
		--msglen;
	}

	if (card_mpuout->lastmidiq == NULL) {
		unsigned long flags;
		/*
		   ** Wait until TX interrupt has occurred.
		   ** This means that the FIFO is empty
		 */
		while (card_mpuout->intr != TRUE);

		spin_lock_irqsave(&card_mpuout->lock, flags);

		while (msglen--) {
			status =
			    hwmpuWriteData
			    (
				    sb_hw,
				    (u8) midimsg
			    );

			if (status != CTSTATUS_SUCCESS) {
				DPD("sblive_mpuoutWriteShortData error!!: byte = ", (u8) midimsg);
				DPD("\nUART BYTE OUT (MISSED) = ", (u8) midimsg);
				msglen++;
			} else {
				DPD("\nUART BYTE OUT = ", (u8) midimsg);
				midimsg >>= 8;
			}
		}

		card_mpuout->intr = FALSE;
		sblive_irqmgrEnableIrq(sb_hw, ENB_TX);
		spin_unlock_irqrestore(&card_mpuout->lock, flags);

		return CTSTATUS_SUCCESS;
	}
	midiq = (struct midi_queue *) kmalloc(sizeof(struct midi_queue), GFP_KERNEL);
	if (midiq == NULL) {
		// message lost
		DPD("sblive_mpuoutWriteShortData error!!: midimsg = ", midimsg);
		return CTSTATUS_NOMEMORY;
	}
	midiq->next = NULL;
	midiq->qtype = 0;
	midiq->length = msglen;
	midiq->sizeLeft = msglen;
	midiq->midibyte = (u8 *) & midiq->refdata;
	midiq->refdata = midimsg;

	spin_lock_irqsave(&card_mpuout->lock, flags);

	if (card_mpuout->firstmidiq == NULL) {
		card_mpuout->lastmidiq = midiq;
		card_mpuout->firstmidiq = midiq;
	} else {
		(card_mpuout->lastmidiq)->next = midiq;
		card_mpuout->lastmidiq = midiq;
	}

	card_mpuout->intr = FALSE;
	sblive_irqmgrEnableIrq(sb_hw, ENB_TX);
	spin_unlock_irqrestore(&card_mpuout->lock, flags);

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//* int    sblive_mpuoutPause(PICARDMIDI pICardMidi,                        *
//*                             u32 handle,                         *
//*                                                                           *
//* Function:   Not Used                                                      *
//*****************************************************************************
int sblive_mpuoutPause(struct sblive_mpuout *card_mpuout, u32 handle)
{
	DPF("sblive_mpuoutPause");
	return CTSTATUS_NOTSUPPORTED;
}


//*****************************************************************************
//* int    sblive_mpuoutStream(PICARDMIDI pICardMidi,                       *
//*                              u32 handle,                        *
//*                              PCARDMIDIBUFFERHDR bufhdr)         *
//*                                                                           *
//* Function:   Not Used                                                      *
//*****************************************************************************
int sblive_mpuoutStream(
			    struct sblive_mpuout *card_mpuout,
			    u32 handle,
			    struct midi_hdr *midihdr
)
{
	DPF("sblive_mpuoutStream");
	return CTSTATUS_NOTSUPPORTED;
}


//*****************************************************************************
//* int    sblive_mpuoutRestart(struct sblive_mpuout *card_mpuout,                      *
//*                               u32 handle,                       *
//*                                                                           *
//* Function:   Not Used                                                      *
//*****************************************************************************
int sblive_mpuoutRestart(struct sblive_mpuout *card_mpuout, u32 handle)
{
	DPF("sblive_mpuoutRestart");
	return CTSTATUS_NOTSUPPORTED;
}


//*****************************************************************************
//* int    sblive_mpuoutReset(PICARDMIDI pICardMidi,                        *
//*                             u32 handle,                         *
//*                                                                           *
//* Function:   Disable the MPU TX.  Return all pending Midi Buffers          *
//*             to client and dispose all pending short msgs.                 *
//*             Then send a note off for all channels and a pedal             *
//*             damper as well.                                               *
//*****************************************************************************
int sblive_mpuoutReset(struct sblive_hw *sb_hw, u32 handle)
{
	struct sblive_mpuout *card_mpuout = sb_hw->card_mpuout;
	int nCh;
	u32 midimsg;
	struct midi_queue *midiq;

	DPF("sblive_mpuoutReset");
	TESTMAGIC(card_mpuout);

	while (card_mpuout->firstmidiq) {
		midiq = card_mpuout->firstmidiq;
		card_mpuout->firstmidiq = midiq->next;
		if (midiq->qtype) {
			u32 callback_msg[3];
			callback_msg[0] = 0;
			callback_msg[1] = midiq->length;
			callback_msg[2] = midiq->refdata;
			card_mpuout->openinfo.CallbackFn
			    (
				    ICARDMIDI_OUTLONGDATA,
				    card_mpuout->openinfo.refdata,
				    (u32) callback_msg
			    );
		}
		kfree(midiq);
	}
	card_mpuout->lastmidiq = NULL;

	// Send a damper pedal off event for all 16 channels
	// BnH 40 00
	for (nCh = 0; nCh < 16; nCh++) {
		midimsg = 0x0040B0 | nCh;
		sblive_mpuoutWriteShortData(sb_hw, handle, midimsg);
	}

	// Send a AllNotesOff event for all 16 channels
	// BnH 78 00
	for (nCh = 0; nCh < 16; nCh++) {
		midimsg = 0x0078B0 | nCh;
		sblive_mpuoutWriteShortData(sb_hw, handle, midimsg);
	}

	// Send a ResetAllControllers event for all 16 channels
	// BnH 79 00
	for (nCh = 0; nCh < 16; nCh++) {
		midimsg = 0x0079B0 | nCh;
		sblive_mpuoutWriteShortData(sb_hw, handle, midimsg);
	}

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//* int    sblive_mpuoutCache(PICARDMIDI pICardMidi,                        *
//*                             u32 handle,                         *
//*                             struct midi_cache * midicache)                  *
//*                                                                           *
//* Function:   Not Used                                                      *
//*****************************************************************************
int sblive_mpuoutCache(
			   struct sblive_mpuout *card_mpuout,
			   u32 handle,
			   struct midi_cache * midicache
)
{
	DPF("sblive_mpuoutCache");
	return CTSTATUS_NOTSUPPORTED;
}


//*****************************************************************************
//* int    sblive_mpuoutGetPosition(PICARDMIDI pICardMidi,                  *
//*                                   u32 handle,                   *
//*                                   u32 *position)                   *
//*                                                                           *
//* Function:   Not Used                                                      *
//*****************************************************************************
int sblive_mpuoutGetPosition(
				 struct sblive_mpuout *card_mpuout,
				 u32 handle,
				 u32 *position
)
{
	DPF("sblive_mpuoutGetPosition");
	return CTSTATUS_NOTSUPPORTED;
}


//*****************************************************************************
//* int    sblive_mpuoutGetControl(struct sblive_mpuout *card_mpuout,                   *
//*                                  u32 handle,                    *
//*                                  u32 controlid,                     *
//*                                  u32 *value)                       *
//*                                                                           *
//* Function:   Not Used                                                      *
//*****************************************************************************
int sblive_mpuoutGetControl(
				struct sblive_mpuout *card_mpuout,
				u32 handle,
				u32 controlid,
				u32 *value
)
{
	DPF("sblive_mpuoutGetControl");

	switch (controlid) {
	case MIDIQUERYACTIVEINST:
		if (card_mpuout == NULL)
			return CTSTATUS_ERROR;
		if (!(card_mpuout->status & FLAGS_AVAILABLE))
			return CTSTATUS_SUCCESS;
		else
			return CTSTATUS_ERROR;

	default:
		break;
	}

	return CTSTATUS_NOTSUPPORTED;
}


//*****************************************************************************
//* int    sblive_mpuoutSetControl(PICARDMIDI pICardMidi,                   *
//*                                  u32 handle,                    *
//*                                  u32 controlid,                     *
//*                                  u32 *value)                       *
//*                                                                           *
//* Function:   Not Used                                                      *
//*****************************************************************************
int sblive_mpuoutSetControl(
				struct sblive_mpuout *card_mpuout,
				u32 handle,
				u32 controlid,
				u32 *value
)
{
	DPF("sblive_mpuoutSetControl");
	return CTSTATUS_NOTSUPPORTED;
}


//*****************************************************************************
//* int sblive_mpuoutDpcCallback ()                                         *
//*****************************************************************************
int sblive_mpuoutDpcCallback(unsigned long refdata, unsigned long param1, unsigned long param2)
{
	int cByteSent = 0;
	int status;
	struct midi_queue *midiq;
	struct midi_queue *doneq = NULL;
	struct sblive_hw *sb_hw = (struct sblive_hw *) refdata;
	struct sblive_mpuout *card_mpuout = sb_hw->card_mpuout;

	while (card_mpuout->firstmidiq) {
		midiq = card_mpuout->firstmidiq;
		while (cByteSent < 4 && midiq->sizeLeft) {
			status =
			    hwmpuWriteData
			    (
				    sb_hw,
				    *midiq->midibyte
			    );

			if (status == CTSTATUS_SUCCESS) {
				++cByteSent;
				--midiq->sizeLeft;
				++midiq->midibyte;
			} else {
				DPF("sblive_mpuoutDpcCallback error!!");
			}
		}
		if (midiq->sizeLeft == 0) {
			if (doneq == NULL)
				doneq = midiq;
			card_mpuout->firstmidiq = midiq->next;
		} else
			break;
	}

	if (card_mpuout->firstmidiq == NULL)
		card_mpuout->lastmidiq = NULL;

	if (doneq) {
		while (doneq != card_mpuout->firstmidiq) {
			u32 callback_msg[3];

			DPD("doneq: ", doneq);

			midiq = doneq;
			doneq = midiq->next;
			if (midiq->qtype) {
				callback_msg[0] = 0;
				callback_msg[1] = midiq->length;
				callback_msg[2] = midiq->refdata;
				card_mpuout->openinfo.CallbackFn
				    (
					    ICARDMIDI_OUTLONGDATA,
					 card_mpuout->openinfo.refdata,
					    (u32) callback_msg
				    );
			} else if (((u8) midiq->refdata) < 0xF0 &&
				   ((u8) midiq->refdata) > 0x7F)
				card_mpuout->laststatus = (u8) midiq->refdata;

			kfree(midiq);
		}
	}
	if (card_mpuout->firstmidiq || cByteSent) {
		card_mpuout->intr = FALSE;
		sblive_irqmgrEnableIrq(sb_hw, ENB_TX);
	}
	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//* int    sblive_mpuoutIrqCallback(unsigned long event,                        *
//*                                   unsigned long refdata,                      *
//*                                   unsigned long param)                        *
//*                                                                           *
//* Function:   IRQ callback handler routine for the MPU out port             *
//*****************************************************************************
int sblive_mpuoutIrqCallback(
				 unsigned long event,
				 unsigned long refdata,
				 unsigned long param
)
{
	struct sblive_hw *sb_hw = (struct sblive_hw *) refdata;
	struct sblive_mpuout *card_mpuout = sb_hw->card_mpuout;

	/*
	   ** Called during ISR. The data & code touched are:
	   **      1. card_mpuout
	   **      2. sblive_irqmgrDisableIrq()
	 */

	DPF("sblive_mpuoutIrqCallback");
	TESTMAGIC(card_mpuout);

	card_mpuout->intr = TRUE;
	sblive_irqmgrDisableIrq(sb_hw, ENB_TX);
	osScheduleDPC(&card_mpuout->dpc);

	return CTSTATUS_SUCCESS;
}
