/*
 **********************************************************************
 *     cardwi.c - PCM input HAL for emu10k1 driver
 *     Copyright 1999, 2000 Creative Labs, Inc.
 *
 **********************************************************************
 *
 *     Date                 Author          Summary of changes
 *     ----                 ------          ------------------
 *     October 20, 1999     Bertrand Lee    base code release
 *
 **********************************************************************
 *
 *     This program is free software; you can redistribute it and/or
 *     modify it under the terms of the GNU General Public License as
 *     published by the Free Software Foundation; either version 2 of
 *     the License, or (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public
 *     License along with this program; if not, write to the Free
 *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 *     USA.
 *
 **********************************************************************
 */

#include "hwaccess.h"
#include "mycommon.h"
#include "halstr.h"
#include "mmwave.h"
#include "cardwi.h"
#include "recmgr.h"


int sblive_waveinIrqCallback(unsigned long, unsigned long, unsigned long);
int sblive_waveinDpcCallback(unsigned long, unsigned long, unsigned long);
int sblive_emuAllocRecBuffer(struct wave_in *, u32 *, u8 **);
int sblive_emuGetRecBuffer(struct sblive_wavein *, struct wave_in *, u32 *, u8 **);
int sblive_emuSetRecBuffer(struct wave_in *, u32, void *);
int sblive_emuDeallocRecBuffer(struct wave_in *);
void initRecordObject(struct sblive_hw *, struct wave_in *);
int sblive_waveinSynchStart(struct sblive_wavein *, u32);

/*************************************************************************** */
/** Function : sblive_waveinInit                                              * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            sb_hw - pointer to hardware object                         * */
/**            mixer - pointer to card mixer                        * */
/**            carddesc - card description                              * */
/**                                                                        * */
/** Return   : CTSTATUS_SUCCESS  -- successful                             * */
/**            CTSTATUS_ERROR    -- failure                                * */
/**                                                                        * */
/** About    : initialize card wave input device.                          * */
/*************************************************************************** */
int sblive_waveinInit(
			  struct sblive_wavein *card_wavein,
			  u8 * carddesc
)
{
	ASSERT(card_wavein);

	/* Init Cardwave Caps */
	card_wavein->caps.product_id = MM_CREATIVE_WAVEIN_PID;
	card_wavein->caps.caps = CARDWAVE_IN;
	card_wavein->caps.controls = 0;
	card_wavein->caps.maxchannels = 2;
	card_wavein->caps.minrate = 8000;
	card_wavein->caps.maxrate = 48000;
	strcpy(card_wavein->caps.wavedesc, carddesc);

	card_wavein->numrecordinst = 0;
	card_wavein->wave_inlist = NULL;

	SETMAGIC(card_wavein);

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinExit                                              * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**                                                                        * */
/** Return   : CTSTATUS_SUCCESS  -- successful                             * */
/**            CTSTATUS_ERROR    -- failure                                * */
/**                                                                        * */
/** About    : exit card wave operation.                                   * */
/*************************************************************************** */
int sblive_waveinExit(struct sblive_wavein *card_wavein)
{
	TESTMAGIC(card_wavein);
	CLEARMAGIC(card_wavein);

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinSetPowerState                                     * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wave in object structure    * */
/**          : powerstate-Powerstate to set the wave object to           * */
/**                                                                        * */
/** Return   : CTSTATUS_SUCCESS  -- successful                             * */
/**            CTSTATUS_ERROR    -- failure                                * */
/**                                                                        * */
/** About    : Sets the powerstate.                                        * */
/*************************************************************************** */
int sblive_waveinSetPowerState(struct sblive_hw *sb_hw, u32 powerstate)
{
	struct sblive_wavein *card_wavein=sb_hw->card_wavein;
	TESTMAGIC(card_wavein);

	switch (powerstate) {
	case POWERSTATE_STANDBY:
	case POWERSTATE_STANDBYRESUME:
		break;

	case POWERSTATE_SUSPEND:
		{
			struct wave_in *wave_in = card_wavein->wave_inlist;
			while (wave_in) {
				if (wave_in->state == CARDWAVE_STATE_STARTED) {
					u32 pending;
					sblive_waveinStop(sb_hw, wave_in, &pending);
					wave_in->state = CARDWAVE_STATE_SUSPEND;
				}
				wave_in = wave_in->next;
			}
		}
		break;

	case POWERSTATE_SUSPENDRESUME:
		{
			struct wave_in *wave_in = card_wavein->wave_inlist;
			while (wave_in) {
				if (wave_in->state == CARDWAVE_STATE_SUSPEND) {
					sblive_waveinStart(sb_hw, wave_in);
				}
				wave_in = wave_in->next;
			}
		}
		break;

	default:
		ASSERT(FALSE);
		return CTSTATUS_ERROR;
	}

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinGetCaps                                           * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            caps - pointer to card wave capabilities object          * */
/**                                                                        * */
/** About    : get card wave device capabilities which is initialized in   * */
/**            sblive_waveinit().                                             * */
/*************************************************************************** */
int sblive_waveinGetCaps(struct sblive_wavein *card_wavein, struct wave_caps *caps)
{
	TESTMAGIC(card_wavein);
	*caps = card_wavein->caps;

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinQueryFormat                                       * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_fmt - pointer to wave format object                  * */
/**            flags - flags that identifies the format to be queried.   * */
/**                                                                        * */
/** About    : query whether a specified wave format is supported by wave  * */
/**            device.                                                     * */
/*************************************************************************** */
int sblive_waveinQueryFormat(
				 struct sblive_wavein *card_wavein,
				 struct wave_format *wave_fmt,
				 u32 flags)
{
	TESTMAGIC(card_wavein);
	ASSERT(wave_fmt);

	if (flags & CARDWAVE_QF_CHANNEL) {
		/* only support one wave input instance */
		if (card_wavein->numrecordinst > 1)
			return CTSTATUS_INUSE;
	}
	if (flags & CARDWAVE_QF_RATE) {
		/* recording */
		/*
		 * sampling rate supported:
		 * 48kHz, 44.1kHz, 32kHz, 24kHz, 22.05kHz, 16kHz, 11.025kHz, 8kHz.
		 */
		if ((wave_fmt->samplingrate != 0xBB80)
		    && (wave_fmt->samplingrate != 0xAC44)
		    && (wave_fmt->samplingrate != 0x7D00)
		    && (wave_fmt->samplingrate != 0x5DC0)
		    && (wave_fmt->samplingrate != 0x5622)
		    && (wave_fmt->samplingrate != 0x3E80)
		    && (wave_fmt->samplingrate != 0x2B11)
		    && (wave_fmt->samplingrate != 0x1F40))
			return CTSTATUS_BADFORMAT_RATE;
	}
	if (flags & CARDWAVE_QF_BITS) {
		/* 16 bit and 8 bit recording are supported */
		if ((wave_fmt->bitspersample != 16)
		    && (wave_fmt->bitspersample != 8))
			return CTSTATUS_BADFORMAT_BITS;
	}
	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinOpen                                              * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_fmt - pointer to wave format object                  * */
/**            CallbackFn - IRQ call back function                         * */
/**            refdata - reference data for call back function           * */
/**            pcallback_size - pointer to the size of data to transfer   * */
/**                              before CallbackFn is called.              * */
/**            numfrags - number of buffer fragments                       * */
/**                                                                        * */
/** Output   : pcallback_size - pointer to the actual call back size.     * */
/**            phandle - pointer to the open handle.                 * */
/**                                                                        * */
/** About    : Open card wave device.                                      * */
/**            1. query whether a specified wave format is supported by    * */
/**               device.                                                  * */
/**            2. allocate emu channel for the specified channel object.   * */
/**            3. attach this wave instance to the channel object list.    * */
/**            4. install wave IRQ handler.                                * */
/**            5. get wave transfer buffer.                                * */
/**            6. get wave instance format.                                * */
/**            7. for recording, install IRQ handler.                      * */
/*************************************************************************** */
int sblive_waveinOpen(
			  struct sblive_hw *sb_hw,
			  struct wave_format *wave_fmt,
			  CALLBACKFN CallbackFn,
			  u32 refdata,
			  u32 *callback_size,
			  u32 numfrags,
			  struct wave_in **handle
)
{
	struct sblive_wavein *card_wavein = sb_hw->card_wavein;
	struct wave_in *wave_in;
	int status;
	u32 bufsiz;
	u8 * buffer;

	ASSERT(wave_fmt);
	TESTMAGIC(card_wavein);


	status = sblive_waveinQueryFormat(card_wavein, wave_fmt, CARDWAVE_QF_ALL);
	if (status != CTSTATUS_SUCCESS) {
		DPF("  sblive_waveinQueryFormat failed.");
		return status;
	}
	/* 8-bit recording directxfer is not supported */
	if ((wave_fmt->flags & CARDWAVE_FORMAT_DO_DIRECTXFER)
	    && (wave_fmt->bitspersample == 8))
		return CTSTATUS_BADFORMAT_BITS;

	wave_in = (struct wave_in *)kmalloc(sizeof(struct wave_in), GFP_KERNEL);

	if (!wave_in) {
		DPF("WAVEININSTOBJ alloc fail");
		return CTSTATUS_NOMEMORY;
	}
#ifdef MEMTEST
	PDEBUG("\ncardwi.c: kmalloc: [%p]\n", wave_in);
#endif

	/* init wave in instance */
	wave_in->next = NULL;
	wave_in->status &= ~FLAGS_AVAILABLE;
	wave_in->state = CARDWAVE_STATE_STOPPED;
	wave_in->synchstart = FALSE;
	wave_in->wave_fmt = *wave_fmt;
	wave_in->CallbackFn = CallbackFn;
	wave_in->emu_voice = NULL;
	wave_in->refdata = refdata;
	wave_in->callbacksize = 0;
	wave_in->process_id = 0;
	wave_in->setpos = FALSE;
	wave_in->position = 0;
	wave_in->timerhandle = 0;
	wave_in->timerinterval = 0;

	/* recording */
	wave_in->rec_ptr = (struct record *) kmalloc(sizeof(struct record), GFP_KERNEL);

	if (!wave_in->rec_ptr) {
		DPF("RECORDOBJ alloc fail");
		kfree((void *) wave_in);
#ifdef MEMTEST
		PDEBUG("\ncardwi.c: kfree: [%p]\n", wave_in);
#endif
		return CTSTATUS_NOMEMORY;
	}
#ifdef MEMTEST
	PDEBUG("\ncardwi.c: kmalloc: [%p]\n", wave_in->rec_ptr);
#endif

	/* init record object */
	initRecordObject(sb_hw, wave_in);

	/* attach this wave instance to the channel object list */
	osListAttach((struct sblive_list **) & card_wavein->wave_inlist, (struct sblive_list *) wave_in);

	if (callback_size != NULL)
		wave_in->callbacksize = *callback_size;
	else
		wave_in->callbacksize = 0;

	if (!(wave_fmt->flags & CARDWAVE_FORMAT_DO_DIRECTXFER)) {

		bufsiz = (wave_in->callbacksize ? wave_in->callbacksize * numfrags : 0xffff);
		status = sblive_emuGetRecBuffer(card_wavein, wave_in, &bufsiz, &buffer);
		if (status != CTSTATUS_SUCCESS) {
			DPF("WaveOpen GetBuffer Fail");
			sblive_waveinClose(sb_hw, wave_in);
			return CTSTATUS_ERROR;
		}
		wave_in->callbacksize = bufsiz / numfrags;
		*callback_size = bufsiz / numfrags;
		/*
		   ** This callback size returned is the size in the play buffer,
		   ** For 8-bit samples, callbacksize of user buffer should be
		   ** half of the callbacksize in play buffer.
		 */
		if (wave_in->wave_fmt.bitspersample == 8)
			*callback_size >>= 1;

	}

	*handle = wave_in;


#if LINUX_VERSION_CODE < 0x020301
	init_waitqueue(&wave_in->wait_queue);
#else
	init_waitqueue_head(&wave_in->wait_queue);
#endif /* LINUX_VERSION_CODE < 0x020301 */
	wave_in->mapped = FALSE;
	wave_in->dsDpc.is_active = FALSE;
	wave_in->dsDpc.refdata = (u32) wave_in;
	wave_in->dsDpc.DPCCallBackFn = sblive_waveinDpcCallback;

	status =
	    sblive_irqmgrInstallIrqHandler
	    (
		    sb_hw,
		    IRQTYPE_RECORD,
		    sblive_waveinIrqCallback,
		    (u32) wave_in
	    );
	if (status != CTSTATUS_SUCCESS) {
		sblive_waveinClose(sb_hw, wave_in);
		return CTSTATUS_ERROR;
	}
	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinClose                                             * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_in  - open handle which is the pointer to the    * */
/**                          specified channel object                      * */
/**                                                                        * */
/** About    : Close wave device.                                          * */
/**            1. deallocate transfer buffer, for playback,                * */
/**               free emu addx space.                                     * */
/**            2. free voice channels.                                     * */
/**            3. uninstall wave IRQ handler.                              * */
/**            4. remove wave instance from channel object list.           * */
/*************************************************************************** */
int sblive_waveinClose(struct sblive_hw *sb_hw, struct wave_in *wave_in)
{
	struct sblive_wavein *card_wavein= sb_hw->card_wavein;
	struct wave_in *tmp;
	u32 dummy;

	ASSERT(wave_in);
	TESTMAGIC(card_wavein);


	if (wave_in->state != CARDWAVE_STATE_STOPPED)
		sblive_waveinStop(sb_hw, wave_in, &dummy);

	tmp = card_wavein->wave_inlist;
	while (tmp) {
		if (tmp == wave_in) {
			if (sblive_emuDeallocRecBuffer(wave_in) != CTSTATUS_SUCCESS)
				ASSERT(FALSE);
			wave_in->status |= FLAGS_AVAILABLE;
			wave_in->state = CARDWAVE_STATE_STOPPED;
			wave_in->synchstart = FALSE;

			sblive_irqmgrUninstallIrqHandler(sb_hw, IRQTYPE_RECORD);
			kfree((void *) wave_in->rec_ptr);

			osListRemove((struct sblive_list **) & card_wavein->wave_inlist, (struct sblive_list *) wave_in);

			/* free channel object which is allocated in sblive_waveOpen(). */
			kfree((void *) wave_in);
#ifdef MEMTEST
			PDEBUG("\ncardwi.c: kfree: [%p]\n", wave_in->rec_ptr);
			PDEBUG("\ncardwi.c: kfree: [%p]\n", wave_in);
#endif
			break;
		}
		tmp = (struct wave_in *) tmp->next;
	}

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_emuGetRecBuffer                                             * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_in - pointer to the specified wavein instance    * */
/**            size - pointer to the size requested                     * */
/**                                                                        * */
/** Output   : size - pointer to the size allocated                     * */
/**            buffer - pointer to the buffer pointer allocated          * */
/**                                                                        * */
/** About    : alloc record buffer.                                        * */
/*************************************************************************** */
int sblive_emuGetRecBuffer(
			   struct sblive_wavein *card_wavein,
			   struct wave_in *wave_in,
			   u32 * size,
			   u8 ** buffer
)
{
	ASSERT(wave_in);
	TESTMAGIC(card_wavein);

	DPF("CARDWAVE : GetBuffer");

	if (!size || !buffer)
		return CTSTATUS_INVALIDPARAM;

	if (*size < wave_in->callbacksize) {
		*size = wave_in->callbacksize;
		return CTSTATUS_INVALIDVALUE;
	}
	/* allocate buffer here */
	if (sblive_emuAllocRecBuffer(wave_in, size, buffer) != CTSTATUS_SUCCESS) {
		DPF("CARDWAVE: allocate buffer fail.");
		return CTSTATUS_ERROR;
	}
	/* recbufsize contains actual record buffer size. */
	*size = wave_in->rec_ptr->recbufsize;

	wave_in->rec_ptr->recbuffer = *buffer;
	wave_in->rec_ptr->recpos = 0;

	return CTSTATUS_SUCCESS;
}

/*************************************************************************** */
/** Function : sblive_waveinGetBuffer                                         * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_in - pointer to the specified wavein instance    * */
/**            size - pointer to the size requested                     * */
/**                                                                        * */
/** Output   : size - pointer to the size allocated                     * */
/**            buffer - pointer to the buffer pointer allocated          * */
/**                                                                        * */
/** About    : get transfer buffer.                                        * */
/*************************************************************************** */
int sblive_waveinGetBuffer(
			       struct sblive_wavein *card_wavein,
			       struct wave_in *wave_in,
			       u32 * size,
			       u8 ** buffer
)
{
	ASSERT(wave_in);
	TESTMAGIC(card_wavein);

	DPF("CARDWAVE : GetBuffer");

	if (!size || !buffer)
		return CTSTATUS_INVALIDPARAM;

	/*
	 * For direct transfer, return pointer.
	 * For non-direct transfer, return actual size and NULL pointer.
	 */
	if (wave_in->wave_fmt.flags & CARDWAVE_FORMAT_DO_DIRECTXFER)
		*buffer = wave_in->rec_ptr->recbuffer;
	else {
		*size = wave_in->rec_ptr->recbufsize;
		*buffer = NULL;
	}

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinSetBuffer                                         * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_in - pointer to the specified wavein instance    * */
/**            size - the buffer size to set                             * */
/**            buffer - pointer to the buffer being set                   * */
/**            flags - CARDWAVE_DMA_BUFFER    (Physically Contiguous)    * */
/**                      CARDWAVE_LINEAR_BUFFER (May not be Physically     * */
/**                                                     Contiguous)        * */
/**                                                                        * */
/** About    : set record buffer, not supported for P8.                    * */
/*************************************************************************** */
int sblive_waveinSetBuffer(struct sblive_hw *sb_hw,
			struct wave_in *wave_in,
			u32 size,
			void *buffer,
			u32 *callback_size,
			u32 flags)
{
	u32 numpages, *PhysAddxTable;

	DPF("CARDWAVE : SetBuffer");
	TESTMAGIC(card_wavein);


	/* recording buffer must be physically continuous */
	if (flags != CARDWAVE_DMA_BUFFER)
		return CTSTATUS_ERROR;

	/* check whether the recording buffer size is supported */
	if (size < 384 || (size > 65536))
		return CTSTATUS_ERROR;
	if ((size % 384) && (size % 448) && (size % 512) && (size % 640))
		return CTSTATUS_ERROR;

	numpages = size / 4096;
	if (size % 4096)
		numpages += 1;

	/* allocate memory for physical addx table, copy page table */
	PhysAddxTable = kmalloc(numpages * sizeof(u32), GFP_KERNEL);

	if (!PhysAddxTable)
		return CTSTATUS_NOMEMORY;

#ifdef MEMTEST
	PDEBUG("\ncardwi.c: kmalloc: [%p]\n", PhysAddxTable);
#endif

	if (osCopyPageTable(buffer, 0, numpages, PhysAddxTable) != CTSTATUS_SUCCESS) {
		kfree((void *) PhysAddxTable);
#ifdef MEMTEST
		PDEBUG("\ncardwi.c: kfree: [%p]\n", PhysAddxTable);
#endif
		return CTSTATUS_ERROR;
	}
	/* check whether the buffer is page-aligned */
	if (PhysAddxTable[0] & 0xfff) {
		kfree((void *) PhysAddxTable);
#ifdef MEMTEST
		PDEBUG("\ncardwi.c: kfree: [%p]\n", PhysAddxTable);
#endif
		return CTSTATUS_ERROR;
	}

	{
		int i, j;
		u32 base_size[4];

		base_size[0] = 384;
		base_size[1] = 448;
		base_size[2] = 512;
		base_size[3] = 640;

		for (i = 0; i < 8; i++)
			for (j = 0; j < 4; j++)
				if (size >= base_size[j]) {
					base_size[j] = base_size[j] * 2;
					wave_in->rec_ptr->bufsizereg = i * 4 + j + 1;
				} else
					goto exitloop;
	exitloop:
		DPD(" bufSizeReg: ", wave_in->rec_ptr->bufsizereg);
	}
	wave_in->rec_ptr->recbuffer = (u8 *) buffer;
	wave_in->rec_ptr->physaddx = PhysAddxTable[0];
	wave_in->rec_ptr->recbufsize = size;
	DPD("   recbufsize: ", wave_in->rec_ptr->recbufsize);

	/* free physical addx table */
	kfree((void *) PhysAddxTable);
#ifdef MEMTEST
	PDEBUG("\ncardwi.c: kfree: [%p]\n", PhysAddxTable);
#endif

	wave_in->timerinterval = (*callback_size)
		* 1000 / wave_in->wave_fmt.samplingrate
		/ (wave_in->wave_fmt.bitspersample / 8)
		/ (wave_in->wave_fmt.channels);

	return CTSTATUS_SUCCESS;
}



/*************************************************************************** */
/** Function : sblive_waveinSynchStart                                        * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**                                                                        * */
/** About    : Synchronize start all wave playback and recording instances * */
/**            in a particular application according to process ID.        * */
/**            Application needs to call SYNCHSTART twice, one is to start * */
/**            all wave playback instances, another to start all wave      * */
/**            recording instances. Because wave out and wave in are       * */
/**            separate devices.                                           * */
/*************************************************************************** */
int sblive_waveinSynchStart(struct sblive_wavein *card_wavein, u32 process_id)
{
	struct wave_in *curr;

	/* start wave recording instances */
	curr = card_wavein->wave_inlist;

	while (curr) {
		if (curr->process_id == process_id) {
			if (curr->wave_fmt.flags & CARDWAVE_FORMAT_DO_DIRECTXFER) {
			}
			if (recmgrStartRecord(curr->rec_ptr) != CTSTATUS_SUCCESS)
				return CTSTATUS_ERROR;
			curr->rec_ptr->prevadcidx = recmgrGetRecIdx(curr->rec_ptr->sb_hw);
		}
		curr = curr->next;
	}

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinGetPosition                                       * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_in - pointer to the specified wavein instance    * */
/**                                                                        * */
/** Output   : position - pointer to current record position            * */
/**                                                                        * */
/** About    : get current record position.                                * */
/*************************************************************************** */
int sblive_waveinGetPosition(struct sblive_wavein *card_wavein,
			  struct wave_in *wave_in,
			  u32 *position)
{
	struct record *rec_ptr = wave_in->rec_ptr;
	u32 curpos;

	TESTMAGIC(card_wavein);

	/* get position of current address, this is in no. of bytes in play buffer */
	if (wave_in->emu_voice != NULL) {
		if (sblive_waveinGetControl(card_wavein, wave_in, WAVECURPOS, &curpos) != CTSTATUS_SUCCESS)
			return CTSTATUS_ERROR;
		curpos *= (rec_ptr->is_stereo + 1);
	} else {
		/* no voices used for recording, get position from wall clock */
		if (wave_in->state == CARDWAVE_STATE_STOPPED)
			curpos = 0;
		else {
			if (recmgrGetPos(wave_in->rec_ptr, &curpos) != CTSTATUS_SUCCESS)
				return CTSTATUS_ERROR;
		}
	}
	*position = curpos;

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinSetPosition                                       * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_in - pointer to the specified wavein instance    * */
/**            position - transfer position to be set                    * */
/**                                                                        * */
/** About    : set current transfer position.                              * */
/*************************************************************************** */
int sblive_waveinSetPosition(struct sblive_wavein *card_wavein,
			  struct wave_in *wave_in,
			  u32 position)
{
	ASSERT(wave_in);
	TESTMAGIC(card_wavein);

	return CTSTATUS_NOTSUPPORTED;
}


/*************************************************************************** */
/** Function : sblive_waveinSetMaxInst                                        * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            value - no. of maximum wave instances to be set           * */
/**                                                                        * */
/** About    : set number of maximum instances of wave device.             * */
/*************************************************************************** */
int sblive_waveinSetMaxInst(struct sblive_wavein *card_wavein, u32 value)
{
	TESTMAGIC(card_wavein);

	return CTSTATUS_NOTSUPPORTED;
}


/*************************************************************************** */
/** Function : sblive_waveinGetProperties                                     * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wave in object structure    * */
/**            prop - pointer to card wave out properties               * */
/**                                                                        * */
/** About    : get sblive_wavein device properties                            * */
/*************************************************************************** */
int sblive_waveinGetProperties(struct sblive_wavein *card_wavein,
			    struct sblive_waveprops *prop)
{
	TESTMAGIC(card_wavein);

	return CTSTATUS_NOTSUPPORTED;
}


/*************************************************************************** */
/** Function : sblive_emuAllocRecBuffer                                           * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            size - pointer to the size requested                     * */
/**                                                                        * */
/** Output   : size - pointer to the size returned                      * */
/**            buffer - pointer to the buffer pointer allocated          * */
/**                                                                        * */
/** About    : allocate buffer for wave transfer.                          * */
/**            recording: a) allocate page-aligned, continous PC memory    * */
/**                          for recording buffer.                         * */
/**                       b) determine start, end, startloop and     * */
/**                          endloop.                                    * */
/*************************************************************************** */
int sblive_emuAllocRecBuffer(struct wave_in *wave_in,
		      u32 * bufsize,
		      u8 ** buffer)
{
	int status;
	u32 numpages, reqsize;
	void *virtaddr;
	unsigned long physaddx;

	ASSERT(wave_in);

	/*
	 * NOTE: record buffer size only can be certain sizes, if the requested
	 *       size is not a nice size, use the smaller nearest size. The minimum
	 *       size is 1k.
	 */
	if (!wave_in->rec_ptr->is16bit)
		*bufsize <<= 1;


	if (*bufsize >= 0x10000) {
		*bufsize = reqsize = 0x10000;
		wave_in->rec_ptr->bufsizereg = 31;
	} else {
		int i, j;
		u32 size[4];

		reqsize = 0;
		size[0] = 384;
		size[1] = 448;
		size[2] = 512;
		size[3] = 640;

		for (i = 0; i < 8; i++)
			for (j = 0; j < 4; j++)
				if (*bufsize >= size[j]) {
					reqsize = size[j];
					size[j] = size[j] * 2;
					wave_in->rec_ptr->bufsizereg = i * 4 + j + 1;
				} else
					goto exitloop;
	      exitloop:
		if (reqsize == 0) {
			reqsize = 384;
			wave_in->rec_ptr->bufsizereg = 1;
		}
		*bufsize = reqsize;
	}
	DPD(" bufSizeReg: ", wave_in->rec_ptr->bufsizereg);

	if (wave_in->rec_ptr->is_stereo) {
		numpages = (reqsize / 8192) * 2;
		if (reqsize % 8192)
			numpages += 2;
	} else {
		numpages = reqsize / 4096;
		if (reqsize % 4096)
			numpages += 1;
	}

	reqsize = numpages * 4096;

	/* recording, recording buffer must be continuous and page-aligned */
	status =
	    osAllocMemPhysical
	    (
		    reqsize,
		    &wave_in->memhandle,
		    &virtaddr,
		    &physaddx
	    );

	if (status != CTSTATUS_SUCCESS)
		return CTSTATUS_NOMEMORY;

	wave_in->rec_ptr->recbuffer = (u8 *) virtaddr;
	wave_in->rec_ptr->physaddx = physaddx;
	wave_in->rec_ptr->recbufsize = *bufsize;
	DPD("   recbufsize: ", wave_in->rec_ptr->recbufsize);

	*buffer = (u8 *) virtaddr;

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_emuSetRecBuffer                                             * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            size - size requested                                     * */
/**            buffer - buffer to set                                     * */
/**                                                                        * */
/** About    : set play buffer for wave transfer.                          * */
/*************************************************************************** */
int sblive_emuSetRecBuffer(struct wave_in *wave_in, u32 size, void *buffer)
{
	ASSERT(wave_in);

	return CTSTATUS_NOTSUPPORTED;
}


/*************************************************************************** */
/** Function : sblive_emuDeallocRecBuffer                                         * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**                                                                        * */
/** About    : deallocate transfer buffer                                  * */
/**            1. for playback, free emu address space.                    * */
/**            2. free PC memory allocated for transfer buffer.            * */
/**            3. clear VioceParam.                                        * */
/*************************************************************************** */
int sblive_emuDeallocRecBuffer(struct wave_in *wave_in)
{
	ASSERT(wave_in);

	if (!wave_in->memhandle)
		return CTSTATUS_ERROR;

	/* free pc memory */
	osFreeMemPhysical(wave_in->memhandle);

	if (wave_in->emu_voice != NULL) {
		struct voice_param * left = &wave_in->emu_voice->voice_params;
		struct voice_param * right = &wave_in->emu_voice->linked_voice->voice_params;
		/* clear VoiceParam */
		left->start = 0;
		left->startloop = 0;
		left->end = 0;
		left->endloop = 0;
		if (wave_in->wave_fmt.channels == 2) {
			right->start = 0;
			right->startloop = 0;
			right->end = 0;
			right->endloop = 0;
		}
	}
	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : initRecordObject                                            * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**                                                                        * */
/** About    : init recording object                                       * */
/*************************************************************************** */
void initRecordObject(struct sblive_hw *sb_hw,struct wave_in *wave_in)
{
	wave_in->rec_ptr->reserved = NULL;
	wave_in->rec_ptr->sb_hw = sb_hw;
	wave_in->rec_ptr->recpos = 0;
	wave_in->rec_ptr->recbufsize = 0;
	wave_in->rec_ptr->recbuffer = NULL;
	wave_in->rec_ptr->physaddx = 0;
	wave_in->rec_ptr->samplingrate = wave_in->wave_fmt.samplingrate;
	wave_in->rec_ptr->is_stereo = (wave_in->wave_fmt.channels == 2) ? 1 : 0;
	wave_in->rec_ptr->is16bit = (wave_in->wave_fmt.bitspersample == 16) ? 1 : 0;
	wave_in->rec_ptr->fSetRecSrc = FALSE;
	wave_in->rec_ptr->prevadcidx = 0;
	wave_in->rec_ptr->pong = FALSE;
}




/*************************************************************************** */
/** Function : sblive_waveinStart                                             * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_in - pointer to the specified wavein instance    * */
/**                                                                        * */
/** About    : start wave in transfer.                                     * */
/**            recording: a) setup voices.                                 * */
/**                       b) set recording source.(CCCA_CC, ADCCR)         * */
/**                       c) enable IRQ.                                   * */
/**                       d) start recording.                              * */
/**                       e) start CA (for get position).                  * */
/*************************************************************************** */
int sblive_waveinStart(struct sblive_hw *sb_hw, struct wave_in *wave_in)
{
	ASSERT(wave_in);
	TESTMAGIC(card_wavein);

	/* if already started, return success */
	if (wave_in->state == CARDWAVE_STATE_STARTED)
		return CTSTATUS_SUCCESS;

	/* recording */
	if (recmgrInit(wave_in->rec_ptr) != CTSTATUS_SUCCESS)
		return CTSTATUS_ERROR;

	/* if recording source is not set yet, use default: AC97 ADC */
	if (!wave_in->rec_ptr->fSetRecSrc) {
		if (recmgrSetControl(wave_in->rec_ptr, WAVERECORD_AC97, 0) != CTSTATUS_SUCCESS)
			return CTSTATUS_SUCCESS;
		wave_in->rec_ptr->fSetRecSrc = TRUE;
	}
	if (sblive_irqmgrEnableIrq(sb_hw, ENB_ADC) != CTSTATUS_SUCCESS)
		return CTSTATUS_ERROR;

	/* actual start */
	if (!wave_in->synchstart) {
		/* recording */
		if (wave_in->wave_fmt.flags & CARDWAVE_FORMAT_DO_DIRECTXFER) {
		}
		if (recmgrStartRecord(wave_in->rec_ptr) != CTSTATUS_SUCCESS)
			return CTSTATUS_ERROR;
		wave_in->rec_ptr->prevadcidx = recmgrGetRecIdx(wave_in->rec_ptr->sb_hw);
	}
	wave_in->state = CARDWAVE_STATE_STARTED;
	wave_in->setpos = FALSE;
	DPF("sblive_wave Started.\n");

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinStop                                              * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_in - pointer to the specified wavein instance    * */
/**            pending - for later use                                  * */
/**                                                                        * */
/** About    : stop wave transfer, disable IRQ.                            * */
/*************************************************************************** */
int sblive_waveinStop(struct sblive_hw *sb_hw,
		   struct wave_in *wave_in,
		   u32 *pending)
{
	struct sblive_wavein *card_wavein=sb_hw->card_wavein;
	u32 dummy;

	ASSERT(wave_in);
	TESTMAGIC(card_wavein);

	sblive_waveinGetXferSize(card_wavein, wave_in, &dummy, pending);

	if (wave_in->emu_voice != NULL) {
		if (sblive_voiceStop(wave_in->emu_voice) != CTSTATUS_SUCCESS)
			return CTSTATUS_ERROR;
	}
	/* recording */
	if (recmgrStopRecord(wave_in->rec_ptr) != CTSTATUS_SUCCESS)
		return CTSTATUS_ERROR;
	if (sblive_irqmgrDisableIrq(sb_hw, ENB_ADC) != CTSTATUS_SUCCESS)
		return CTSTATUS_ERROR;

	if (wave_in->emu_voice != NULL) {
		struct voice_param *left = &wave_in->emu_voice->voice_params;
		left->startloop = left->start;
		if (wave_in->rec_ptr->is_stereo) {
			struct voice_param *right = &wave_in->emu_voice->linked_voice->voice_params;

			right->startloop = right->start;
		}
	}
	wave_in->rec_ptr->fSetRecSrc = FALSE;
	wave_in->rec_ptr->recpos = 0;
	wave_in->rec_ptr->pong = FALSE;
	wave_in->state = CARDWAVE_STATE_STOPPED;
	wave_in->process_id = 0;

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinGetXferSize                                       * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_in - pointer to the specified wavein instance    * */
/**                                                                        * */
/** Output   : size - pointer to the transfer size                      * */
/**            pending - pointer to the size to be transfered           * */
/**                                                                        * */
/** About    : get the size of data in bytes that the specified wave device* */
/**            can process at the time of this function is called.         * */
/**                                                                        * */
/** Note     : this transfer size returned is referring to user buffer.    * */
/*************************************************************************** */
int sblive_waveinGetXferSize(
				 struct sblive_wavein *card_wavein,
				 struct wave_in *wave_in,
				 u32 *size,
				 u32 *pending
)
{
	struct record *rec_ptr = wave_in->rec_ptr;
	u32 curpos;

	ASSERT(wave_in);
	TESTMAGIC(card_wavein);

	/* get position of current address, this is in no. of bytes in play buffer */
	if (wave_in->emu_voice != NULL) {
		if (sblive_waveinGetControl(card_wavein, wave_in, WAVECURPOS, &curpos) != CTSTATUS_SUCCESS) {
			DPF(" sblive_waveinGetControl failed");
			return CTSTATUS_ERROR;
		}
		curpos *= (rec_ptr->is_stereo + 1);
	} else {
		/* no voices used for recording, get position from wall clock */
		if (wave_in->state == CARDWAVE_STATE_STOPPED)
			curpos = 0;
		else {
			if (recmgrGetPos(wave_in->rec_ptr, &curpos) != CTSTATUS_SUCCESS) {
				DPF(" recmgrGetPos failed");
				return CTSTATUS_ERROR;
			}
		}
	}

	/* recpos is the actual position in user buffer and play buffer */
	if (curpos >= rec_ptr->recpos)
		*size = curpos - rec_ptr->recpos;
	else
		*size = rec_ptr->recbufsize
		    - (rec_ptr->recpos - curpos);

#if defined(EMUA0)
	/*
	 * Only emu8010 A0 silicon needs this workaround,
	 * A1 silicon, take out this part.
	 */
	if (*size > 32)
		*size -= 32;
#endif

	if (!rec_ptr->is16bit)
		*size >>= 1;

	*pending = 0;

	//   DPD("  CA:",curpos);
	//    DPD("  RecPos:",rec_ptr->recpos);
	//    DPD("  get XferSize:",*size);

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinXferData                                          * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_in - pointer to the specified wavein instance    * */
/**            Data - pointer to the data to be transfered                * */
/**            size - data size to be transfered(size in user buffer,   * */
/**                      for 8-bit sample, this is different from play     * */
/**                      buffer.                                           * */
/**                                                                        * */
/** Output   : size - data size transfered                              * */
/**                                                                        * */
/** About    : transfer the data to/from the wave device.                  * */
/*************************************************************************** */
int sblive_waveinXferData(struct sblive_wavein *card_wavein,
		       struct wave_in *wave_in,
		       u8 *data,
		       u32 *size)
{
	struct record *rec_ptr = wave_in->rec_ptr;
	u32 sizeToCopy, sizeToCopyNow, sizeCopied;

	ASSERT(wave_in);
	TESTMAGIC(card_wavein);

	if (!data || !size) {
		return CTSTATUS_INVALIDPARAM;
	}
	DPD("  XferData, size request to xfer ", *size);

	sizeToCopy = min(rec_ptr->recbufsize * (rec_ptr->is16bit + 1) / 2, *size);
	sizeCopied = 0;

	if (!sizeToCopy) {
		*size = 0;
		return CTSTATUS_SUCCESS;
	}
	while (sizeCopied < sizeToCopy) {
		sizeToCopyNow = sizeToCopy - sizeCopied;
		sizeToCopyNow = min(sizeToCopyNow,
				      (rec_ptr->recbufsize - rec_ptr->recpos) * (rec_ptr->is16bit + 1) / 2);
		if (rec_ptr->is16bit)
			copy_to_user(&data[sizeCopied],
			     &rec_ptr->recbuffer[rec_ptr->recpos],
				     sizeToCopyNow);
		else {
			u8 *DstBuf = &data[sizeCopied];
			u16 *SrcBuf = (u16 *) & rec_ptr->recbuffer[rec_ptr->recpos];
			u32 count = sizeToCopyNow;

			while (count--) {
				long lSample;
				u8 bSample;

				lSample = (short) *SrcBuf++;
#ifdef RECTEST
				DPD(" lSample -> ", lSample);
#endif
				lSample += 128;
				if (lSample > 32767)
					lSample = 32767;
				bSample = (u8) ((lSample >> 8) ^ 0x80);
				copy_to_user(DstBuf, &bSample, 1);
				DstBuf++;
			}
		}

		rec_ptr->recpos += sizeToCopyNow * 2 / (rec_ptr->is16bit + 1);
		rec_ptr->recpos %= rec_ptr->recbufsize;
		sizeCopied += sizeToCopyNow;
	}
	*size = sizeToCopy;
	DPD("       size copied ", *size);

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinFillSilence                                       * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_in - pointer to the specified wavein instance    * */
/**            size - data size to be filled                            * */
/**                                                                        * */
/** Output   : size - data size filled                                  * */
/**                                                                        * */
/** About    : fill silent data to play buffer,only used for wave output.  * */
/*************************************************************************** */
int sblive_waveinFillSilence(struct sblive_wavein *card_wavein,
			  struct wave_in *wave_in,
			  u32 *size)
{
	ASSERT(wave_in);
	TESTMAGIC(card_wavein);

	return CTSTATUS_NOTSUPPORTED;
}



/*************************************************************************** */
/** Function : sblive_waveinSetControl                                        * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_in - pointer to the specified wavein instance    * */
/**            ctrlid - control type                                     * */
/**            value - the value of the specified control to be set      * */
/**                                                                        * */
/** About    : set the specified control value of the wave device.         * */
/*************************************************************************** */
int sblive_waveinSetControl(struct sblive_wavein *card_wavein,
			 struct wave_in *wave_in,
			 u32 ctrlid,
			 u32 *value)
{
	u32 val;

	TESTMAGIC(card_wavein);

	if (!card_wavein)
		return CTSTATUS_ERROR;

	if (value)
		val = *value;
	else
		val = 0;

	switch (ctrlid) {
	case WAVEOBJVOLUME:
	case WAVEOBJMUTE:
	case WAVEOBJREVERB:
	case WAVEOBJCHORUS:
		return CTSTATUS_NOTSUPPORTED;

	case WAVESYNCHSTART:
		if (sblive_waveinSynchStart(card_wavein, val) != CTSTATUS_SUCCESS)
			return CTSTATUS_ERROR;
		return CTSTATUS_SUCCESS;

	default:
		break;
	}

	if (!wave_in)
		return CTSTATUS_ERROR;

	switch (ctrlid) {
	case WAVEINSTANCEVOLUME:
	case WAVEINSTANCEREVERB:
	case WAVEINSTANCECHORUS:
		return CTSTATUS_NOTSUPPORTED;

	case WAVECURPOS:
		/* recording set position not supported */
	case WAVESETFREQUENCY:
	case WAVESTARTLOOP:
		return CTSTATUS_NOTSUPPORTED;

	case WAVEWRITEPOINTER:
		/*
		 * For wave input device, set the offset of write pointer.
		 *
		 * NOTE!!!
		 * Later, if DirectSound can support wave recording, this
		 * should be changed.
		 */
		wave_in->rec_ptr->recpos += val * 2 / (wave_in->rec_ptr->is16bit + 1);
		wave_in->rec_ptr->recpos %= wave_in->rec_ptr->recbufsize;
		break;

	case WAVERECORD_AC97:
		/* record from AC97 ADC */
		if (recmgrSetControl(wave_in->rec_ptr, ctrlid, val) != CTSTATUS_SUCCESS)
			return CTSTATUS_ERROR;
		wave_in->rec_ptr->fSetRecSrc = TRUE;
		break;

	case WAVERECORD_E8K:
		{
			struct voice_cntlset setting[1];

			/* record from the output of emu8000 */
			setting[0].paramID = CCCA_CNTL;
			setting[0].value = 0x0e;	// left SAA

			if (sblive_voiceSetControl(wave_in->emu_voice, setting, 1) != CTSTATUS_SUCCESS)
				return CTSTATUS_ERROR;

			if (wave_in->rec_ptr->is_stereo && wave_in->emu_voice->linked_voice) {
				setting[0].paramID = CCCA_CNTL;
				setting[0].value = 0x0f;	// right SAA

				if (sblive_voiceSetControl(wave_in->emu_voice->linked_voice, setting, 1) != CTSTATUS_SUCCESS)
					return CTSTATUS_ERROR;
			}
			if (recmgrSetControl(wave_in->rec_ptr, ctrlid, val) != CTSTATUS_SUCCESS)
				return CTSTATUS_ERROR;
			wave_in->rec_ptr->fSetRecSrc = TRUE;
			break;
		}
	case WAVESYNCHSETUP:
		wave_in->process_id = val;
		wave_in->synchstart = TRUE;
		break;

	default:
		ASSERT(FALSE);
		return CTSTATUS_ERROR;
	}

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinGetControl                                        * */
/**                                                                        * */
/** Input    : card_wavein - pointer to card wavein object               * */
/**            wave_in - pointer to the specified wavein instance    * */
/**            ctrlid - control type                                     * */
/**                                                                        * */
/** Output   : value - pointer to the control value                     * */
/**                                                                        * */
/** About    : get the specified control value of the wave device.         * */
/*************************************************************************** */
int sblive_waveinGetControl(struct sblive_wavein *card_wavein,
			 struct wave_in *wave_in,
			 u32 ctrlid,
			 u32 *value)
{
	TESTMAGIC(card_wavein);

	switch (ctrlid) {
	case WAVEOBJVOLUME:
	case WAVEOBJREVERB:
	case WAVEOBJCHORUS:
		return CTSTATUS_NOTSUPPORTED;

	case WAVEQUERYACTIVEINST:
		if (card_wavein->wave_inlist != NULL)
			return CTSTATUS_SUCCESS;
		else
			return CTSTATUS_ERROR;

	default:
		break;
	}

	if (!wave_in)
		return CTSTATUS_ERROR;

	switch (ctrlid) {
	case WAVEINSTANCEVOLUME:
	case WAVEINSTANCEREVERB:
	case WAVEINSTANCECHORUS:
		return CTSTATUS_NOTSUPPORTED;

	case WAVECURPOS:
		{
			u32 curpos;

			if (sblive_voiceGetControl(wave_in->emu_voice,
			     CCCA_CA, value) != CTSTATUS_SUCCESS)
				return CTSTATUS_ERROR;

			curpos = *value;
//                DPD("getcontrol: CA is", curpos);

			/* there is no actual start yet */
			if (wave_in->state == CARDWAVE_STATE_STOPPED)
				*value = 0;
			else {
				*value = curpos - wave_in->emu_voice->voice_params.start;
				*value <<= wave_in->rec_ptr->is16bit;
			}
//               DPD("getcontrol: position is", *value);
		}
		break;

	case WAVESTARTLOOP:

		if (sblive_voiceGetControl(wave_in->emu_voice,
			     PSST_ST, value) != CTSTATUS_SUCCESS)
			return CTSTATUS_ERROR;

		*value -= wave_in->emu_voice->voice_params.start;
		*value <<= wave_in->rec_ptr->is16bit;
		break;

	case WAVEENDLOOP:

		if (sblive_voiceGetControl(wave_in->emu_voice,
			       CSL_L, value) != CTSTATUS_SUCCESS)
			return CTSTATUS_ERROR;
		*value -= wave_in->emu_voice->voice_params.start;
		*value <<= wave_in->rec_ptr->is16bit;
		break;

	case WAVEWRITEPOINTER:
		/* get write pointer for a particular wave instance */
		*value = wave_in->rec_ptr->recpos;
		break;

	default:
		ASSERT(FALSE);
		return CTSTATUS_ERROR;
	}

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/** Function : sblive_waveinIrqCallback                                       * */
/**                                                                        * */
/** Input    : event - event that cause the callback                     * */
/**            refdata - reference data for this callback                * */
/**            param - parameter used for this callback                  * */
/**                                                                        * */
/** About    : wave IRQ callback function.                                 * */
/*************************************************************************** */
int sblive_waveinIrqCallback(unsigned long event, unsigned long refdata, unsigned long param)
{
	struct wave_in *wave_in = (struct wave_in *) refdata;

	if (wave_in->state == CARDWAVE_STATE_STOPPED)
		return CTSTATUS_SUCCESS;

	if (event == VOICECALLBACK_EVENT_CALLBACKSIZEREACHED) {
		//        DPF("   Recording IRQ!");

		wave_in->rec_ptr->prevadcidx = recmgrGetRecIdx(wave_in->rec_ptr->sb_hw);
		wave_in->rec_ptr->pong ^= 1;

		/* for directxfer, don't need to do irq callback */
		if (!(wave_in->wave_fmt.flags
		      & CARDWAVE_FORMAT_DO_DIRECTXFER)) {
			if (wave_in->CallbackFn != NULL) {
				if (!wave_in->dsDpc.is_active) {
					wave_in->dsDpc.is_active = TRUE;
					osScheduleDPC(&wave_in->dsDpc);
				}
			}
		}
		return CTSTATUS_SUCCESS;
	}
	return CTSTATUS_INVALIDPARAM;
}


int sblive_waveinDpcCallback(unsigned long refdata, unsigned long param1, unsigned long param2)
{
	struct wave_in *wave_in = (struct wave_in *) refdata;

	wave_in->dsDpc.is_active = FALSE;

	if (wave_in->CallbackFn != NULL)
		wave_in->CallbackFn(VOICECALLBACK_EVENT_CALLBACKSIZEREACHED, wave_in->refdata, 0);

	return CTSTATUS_SUCCESS;
}
