/*     
 **********************************************************************
 *     ecard.c - E-card initialization code
 *     Copyright 1999, 2000 Creative Labs, Inc. 
 * 
 ********************************************************************** 
 * 
 *     Date                 Author          Summary of changes 
 *     ----                 ------          ------------------ 
 *     October 20, 1999     Bertrand Lee    base code release 
 *     November 2, 1999     Alan Cox        cleaned up
 * 
 ********************************************************************** 
 * 
 *     This program is free software; you can redistribute it and/or 
 *     modify it under the terms of the GNU General Public License as 
 *     published by the Free Software Foundation; either version 2 of 
 *     the License, or (at your option) any later version. 
 * 
 *     This program is distributed in the hope that it will be useful, 
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *     GNU General Public License for more details. 
 * 
 *     You should have received a copy of the GNU General Public 
 *     License along with this program; if not, write to the Free 
 *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, 
 *     USA. 
 * 
 ********************************************************************** 
 */

#include "hwaccess.h"
#include "mycommon.h"


/*************************************************************************
 * ECARD functional implementation
 *************************************************************************/

/* In A1 Silicon, these bits are in the HC register */
#define HOOKN_BIT   (1L << 12)
#define HANDN_BIT   (1L << 11)
#define PULSEN_BIT  (1L << 10)

#define EC_GDI1 (1 << 13)
#define EC_GDI0 (1 << 14)

#define EC_NUM_CONTROL_BITS 20

#define EC_AC3_DATA_SELN  0x0001L
#define EC_EE_DATA_SEL    0x0002L
#define EC_EE_CNTRL_SELN  0x0004L
#define EC_EECLK          0x0008L
#define EC_EECS           0x0010L
#define EC_EESDO          0x0020L
#define EC_TRIM_CSN	  0x0040L
#define EC_TRIM_SCLK	  0x0080L
#define EC_TRIM_SDATA	  0x0100L
#define EC_TRIM_MUTEN	  0x0200L
#define EC_ADCCAL	  0x0400L
#define EC_ADCRSTN	  0x0800L
#define EC_DACCAL	  0x1000L
#define EC_DACMUTEN	  0x2000L
#define EC_LEDN		  0x4000L

#define EC_SPDIF0_SEL_SHIFT	15
#define EC_SPDIF1_SEL_SHIFT	17
#define EC_SPDIF0_SEL_MASK	(0x3L << EC_SPDIF0_SEL_SHIFT)
#define EC_SPDIF1_SEL_MASK	(0x7L << EC_SPDIF1_SEL_SHIFT)
#define EC_SPDIF0_SELECT(_x) (((_x) << EC_SPDIF0_SEL_SHIFT) & EC_SPDIF0_SEL_MASK)
#define EC_SPDIF1_SELECT(_x) (((_x) << EC_SPDIF1_SEL_SHIFT) & EC_SPDIF1_SEL_MASK)
#define EC_CURRENT_PROM_VERSION 0x01	/* Self-explanatory.  This should
					   * be incremented any time the EEPROM's
					   * format is changed.  */

#define EC_EEPROM_SIZE	        0x40	/* ECARD EEPROM has 64 16-bit words */

/* Addresses for special values stored in to EEPROM */
#define EC_PROM_VERSION_ADDR 0x20	/* Address of the current prom version */
#define EC_BOARDREV0_ADDR    0x21	/* LSW of board rev */
#define EC_BOARDREV1_ADDR    0x22	/* MSW of board rev */

#define EC_LAST_PROMFILE_ADDR 0x2f

#define EC_SERIALNUM_ADDR    0x30	/* First word of serial number.  The 
					   * can be up to 30 characters in length
					   * and is stored as a NULL-terminated
					   * ASCII string.  Any unused bytes must be
					   * filled with zeros */
#define EC_CHECKSUM_ADDR     0x3f	/* Location at which checksum is stored */



/* Most of this stuff is pretty self-evident.  According to the hardware 
 * dudes, we need to leave the ADCCAL bit low in order to avoid a DC 
 * offset problem.  Weird.
 */
#define EC_RAW_RUN_MODE	(EC_DACMUTEN | EC_ADCRSTN | EC_TRIM_MUTEN | EC_TRIM_CSN)


#define EC_DEFAULT_ADC_GAIN   0xC4C4
#define EC_DEFAULT_SPDIF0_SEL 0x0
#define EC_DEFAULT_SPDIF1_SEL 0x4

#define HC_EA 0x01L

/* ECARD state structure.  This structure maintains the state
 * for various portions of the the ECARD's onboard hardware.
 */
struct ecard_state {
	u32 controlbits;
	u16 ADCgain;
	u16 mux0setting;
	u16 mux1setting;
	u16 mux2setting;
};


/* Private routines */
static void _ECARDsetadcgain(struct sblive_hw *, struct ecard_state *, u16 value);
static void _ECARDwrite(struct sblive_hw *, u32);

/**************************************************************************
 * @func Set the gain of the ECARD's CS3310 Trim/gain controller.  The
 * trim value consists of a 16bit value which is composed of two
 * 8 bit gain/trim values, one for the left channel and one for the
 * right channel.  The following table maps from the Gain/Attenuation
 * value in decibels into the corresponding bit pattern for a single
 * channel.
 */

static void _ECARDsetadcgain(struct sblive_hw * sb_hw, struct ecard_state * pecard, u16 gain)
{
	u32 bit;
	pecard->ADCgain = gain;

	/* Enable writing to the TRIM registers */
	_ECARDwrite(sb_hw, pecard->controlbits & ~EC_TRIM_CSN);

	/* Do it again to insure that we meet hold time requirements */
	_ECARDwrite(sb_hw, pecard->controlbits & ~EC_TRIM_CSN);

	for (bit = (1L << 15); bit; bit >>= 1) {

		u32 value = pecard->controlbits & ~(EC_TRIM_CSN | EC_TRIM_SDATA);

		if (gain & bit)
			value |= EC_TRIM_SDATA;

		/* Clock the bit */
		_ECARDwrite(sb_hw, value);
		_ECARDwrite(sb_hw, value | EC_TRIM_SCLK);
		_ECARDwrite(sb_hw, value);
	}

	_ECARDwrite(sb_hw, pecard->controlbits);
}

/**************************************************************************
 * @func Clock bits into the Ecard's control latch.  The Ecard uses a
 *  control latch will is loaded bit-serially by toggling the Modem control
 *  lines from function 2 on the E8010.  This function hides these details
 *  and presents the illusion that we are actually writing to a distinct
 *  register.
 */

static void _ECARDwrite(struct sblive_hw * sb_hw, u32 value)
{
	u16 count;
	u32 data;
	u32 hcval;

	hcval = inl(sb_hw->hwconfigaddx) & ~(HOOKN_BIT | HANDN_BIT | PULSEN_BIT);

	outl(hcval, sb_hw->hwconfigaddx);

	for (count = 0; count < EC_NUM_CONTROL_BITS; count++) {

		/* Set up the value */
		data = ((value & 0x1) ? PULSEN_BIT : 0);
		value >>= 1;

		outl(hcval | data, sb_hw->hwconfigaddx);

		/* Clock the shift register */
		outl(hcval | data | HANDN_BIT, sb_hw->hwconfigaddx);
		outl(hcval | data, sb_hw->hwconfigaddx);
	}

	/* Latch the bits */
	outl(hcval | HOOKN_BIT, sb_hw->hwconfigaddx);
	outl(hcval, sb_hw->hwconfigaddx);
}

int sblive_ecard_init(struct sblive_hw * sb_hw)
{
	u32 hcval;
	struct ecard_state ecard;

	/* Set up the initial settings */
	ecard.mux0setting = EC_DEFAULT_SPDIF0_SEL;
	ecard.mux1setting = EC_DEFAULT_SPDIF1_SEL;
	ecard.mux2setting = 0;
	ecard.ADCgain = EC_DEFAULT_ADC_GAIN;
	ecard.controlbits = EC_RAW_RUN_MODE |
	    EC_SPDIF0_SELECT(ecard.mux0setting) |
	    EC_SPDIF1_SELECT(ecard.mux1setting);


	/* Step 0: Set the codec type in the hardware control register 
	 * and enable audio output */
	hcval = inl(sb_hw->hwconfigaddx);
	outl(hcval | HC_EA | 0x10000L, sb_hw->hwconfigaddx);
	inl(sb_hw->hwconfigaddx);

	/* Step 1: Turn off the led and deassert TRIM_CS */
	_ECARDwrite(sb_hw, EC_ADCCAL | EC_LEDN | EC_TRIM_CSN);

	/* Step 2: Calibrate the ADC and DAC */
	_ECARDwrite(sb_hw, EC_DACCAL | EC_LEDN | EC_TRIM_CSN);

	/* Step 3: Wait for awhile;   XXX We can't get away with this
	 * under a real operating system; we'll need to block and wait that
	 * way. */
	halWC_WAIT(sb_hw, 48000);

	/* Step 4: Switch off the DAC and ADC calibration.  Note
	 * That ADC_CAL is actually an inverted signal, so we assert
	 * it here to stop calibration.  */
	_ECARDwrite(sb_hw, EC_ADCCAL | EC_LEDN | EC_TRIM_CSN);

	/* Step 4: Switch into run mode */
	_ECARDwrite(sb_hw, ecard.controlbits);

	/* Step 5: Set the analog input gain */
	_ECARDsetadcgain(sb_hw, &ecard, ecard.ADCgain);

	return CTSTATUS_SUCCESS;
}
