/*     
 **********************************************************************
 *     main.c - Creative EMU10K1 audio driver
 *     Copyright 1999, 2000 Creative Labs, Inc. 
 * 
 ********************************************************************** 
 * 
 *     Date                 Author          Summary of changes 
 *     ----                 ------          ------------------ 
 *     October 20, 1999     Bertrand Lee    base code release 
 *     November 2, 1999     Alan Cox        cleaned up stuff
 * 
 ********************************************************************** 
 * 
 *     This program is free software; you can redistribute it and/or 
 *     modify it under the terms of the GNU General Public License as 
 *     published by the Free Software Foundation; either version 2 of 
 *     the License, or (at your option) any later version. 
 * 
 *     This program is distributed in the hope that it will be useful, 
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *     GNU General Public License for more details. 
 * 
 *     You should have received a copy of the GNU General Public 
 *     License along with this program; if not, write to the Free 
 *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, 
 *     USA. 
 * 
 ********************************************************************** 
 *
 *      Supported devices:
 *      /dev/dsp:     Standard /dev/dsp device, OSS-compatible
 *      /dev/mixer:   Standard /dev/mixer device, OSS-compatible
 *      /dev/sndstat: Standard /dev/sndstat device, mostly OSS-compatible
 *      /dev/midi:    Raw MIDI UART device, mostly OSS-compatible
 *
 *      Revision history:
 *      0.1 beta Initial release
 *      0.2 Lowered initial mixer vol. Improved on stuttering wave playback. Added MIDI UART support.
 *      0.3 Fixed mixer routing bug, added APS, joystick support.
 *      0.4 Added rear-channel, SPDIF support. 
 *
 ********************************************************************** 
 */


// These are only included once per module
//#include <linux/version.h>
#define __NO_VERSION__
#include <linux/module.h>
//#include <linux/kerneld.h>

#include "hwaccess.h"
#include "mycommon.h"
#include "cardwo.h"

// Macros
#if LINUX_VERSION_CODE >= 0x020311
#define RSRCISIOREGION(dev,num) ((dev)->resource[(num)].start != 0 && \
                                 ((dev)->resource[(num)].flags & PCI_BASE_ADDRESS_SPACE) == PCI_BASE_ADDRESS_SPACE_IO)
#define RSRCADDRESS(dev,num) ((dev)->resource[(num)].start)
#endif /* LINUX_VERSION_CODE >= 0x020311 */

// Defines
#ifndef PCI_VENDOR_ID_CREATIVE
#define PCI_VENDOR_ID_CREATIVE 0x1102
#endif

#ifndef PCI_DEVICE_ID_CREATIVE_EMU10K1
#define PCI_DEVICE_ID_CREATIVE_EMU10K1 0x0002
#endif

#define EMU10K1_EXTENT	0x20	// 32 byte I/O space
#define JOY_EXTENT	0x8	// 8 byte I/O space
#define APS_SERIAL	0x40011102
#define MAX_EMU10K1	5

// Global var instantiation
static char kernel_version[] = UTS_RELEASE;
struct sblive_hw *sblive_devs = NULL;
#if LINUX_VERSION_CODE >= 0x020100
static unsigned long joystick[MAX_EMU10K1 + 1] =
{0,};
#endif

// Function prototypes
extern int audio_init(struct sblive_hw *sb_hw, s8 *pname);
extern int audio_exit(struct sblive_hw *sb_hw);
extern int sblive_mixer_init(struct sblive_hw *sb_hw);
extern void sndstat_init(void);
extern int midi_init(struct sblive_hw *sb_hw);
extern int midi_exit(struct sblive_hw *sb_hw);
extern int sblive_ecard_init(struct sblive_hw *sb_hw);

#if LINUX_VERSION_CODE < 0x020100
/* Utility functions */
inline int copy_from_user(void *to, const void *from, unsigned long n)
{
	int i = verify_area(VERIFY_READ, from, n);
	if (i)
		return i;
	memcpy_fromfs(to, from, n);
	return 0;
}

inline int copy_to_user(void *to, const void *from, unsigned long n)
{
	int i = verify_area(VERIFY_WRITE, to, n);
	if (i)
		return i;
	memcpy_tofs(to, from, n);
	return 0;
}
#endif				// LINUX_VERSION_CODE


// Driver initialization routine
#ifdef MODULE
int init_module(void)
#else
int __init init_emu10k1(void)
#endif
{
	struct sblive_hw *hw_obj;
	u32 hwflags = 0;
	unsigned int index = 0, serial = 0;
	unsigned long ioaddr;
	unsigned char bus, devfn, ucRevID = 0;
	struct sblive_config dsConfig;
	struct sblive_config *config = &dsConfig;
	unsigned long irq;
#if LINUX_VERSION_CODE >= 0x020100
	struct pci_dev *pcidev = NULL, *pcidev_joy = NULL, *pci_tmp = NULL;
	unsigned char devfn1;
	unsigned long ioaddr1 = 0;
#endif

	PDEBUG("init_module() called\n");

	// Do not export symbols 
#ifdef MODULE
#if LINUX_VERSION_CODE < 0x020100
	register_symtab(NULL);
#else
	EXPORT_NO_SYMBOLS;
#endif
#endif				// MODULE

#if LINUX_VERSION_CODE < 0x020100
	if (!pcibios_present())
		return -ENODEV;

	// PCI BIOS detected
	for (index = 0; index < MAX_EMU10K1; index++) {
		int nResult;
		nResult = pcibios_find_device(PCI_VENDOR_ID_CREATIVE,
					  PCI_DEVICE_ID_CREATIVE_EMU10K1,
					      index,
					      &bus,
					      &devfn);
		if (nResult != PCIBIOS_SUCCESSFUL)
			break;

		// Device found - Query PCI BIOS info
		PDEBUG("\nEmu10k1 device %d found!\n", index);

		pcibios_read_config_dword(bus, devfn, PCI_BASE_ADDRESS_0, &ioaddr);
		if (ioaddr == 0 || (ioaddr & PCI_BASE_ADDRESS_SPACE) != PCI_BASE_ADDRESS_SPACE_IO)
			continue;
		ioaddr = ioaddr & PCI_BASE_ADDRESS_IO_MASK;
		PDEBUG("IO Base Address = %#x\n", ioaddr);

		pcibios_read_config_byte(bus, devfn, PCI_INTERRUPT_LINE, &irq);
		if (irq == 0 || irq == 0xff)
			continue;
		PDEBUG("IRQ Line = %u\n", irq);

		pcibios_read_config_byte(bus, devfn, PCI_REVISION_ID, &ucRevID);
		PDEBUG("Revision ID = %u\n", ucRevID);

		// Kernel source bug - Should actually be VENDOR_ID 
		pcibios_read_config_dword(bus, devfn, PCI_SUBSYSTEM_ID, &serial);
		PDEBUG("Serial No = %08x\n", serial);
#else
	if (!pci_present())
		return -ENODEV;

	// PCI BIOS detected
	for (index = 0; index < MAX_EMU10K1 &&
			(pcidev = pci_find_device(PCI_VENDOR_ID_CREATIVE, PCI_DEVICE_ID_CREATIVE_EMU10K1, pcidev));
			index++) 
	{
		bus = pcidev->bus->number;
		devfn = pcidev->devfn;
		PDEBUG("\nFunction 0: devfn %x\n", devfn);
		pci_tmp = pcidev->next;
		while (pci_tmp) {
			devfn1 = pci_tmp->devfn;
			if (devfn1 == devfn + 1) {
				PDEBUG("\nFunction 1 exists: devfn %x\n", devfn1);
				pcidev_joy = pci_tmp;
				break;
			}
			pci_tmp = pci_tmp->next;
		}
		// Device found - Query PCI BIOS info
		PDEBUG("\nEmu10k1 device %d found!\n", index);

#if LINUX_VERSION_CODE < 0x020311
		ioaddr = pcidev->base_address[0];
#else
		ioaddr = RSRCADDRESS(pcidev, 0);
#endif /* LINUX_VERSION_CODE < 0x020311 */
		if (pcidev_joy)
#if LINUX_VERSION_CODE < 0x020311
			ioaddr1 = pcidev_joy->base_address[0];
#else
			ioaddr1 = RSRCADDRESS(pcidev_joy, 0);
#endif /* LINUX_VERSION_CODE < 0x020311 */

#if LINUX_VERSION_CODE < 0x020311
		if (ioaddr == 0 || (ioaddr & PCI_BASE_ADDRESS_SPACE) != PCI_BASE_ADDRESS_SPACE_IO
		    || (ioaddr1 == 0) || (ioaddr1 & PCI_BASE_ADDRESS_SPACE) != PCI_BASE_ADDRESS_SPACE_IO)
#else
		if (!RSRCISIOREGION(pcidev, 0) || !RSRCISIOREGION(pcidev_joy, 0))
#endif /* LINUX_VERSION_CODE < 0x020311 */
			continue;
		ioaddr = ioaddr & PCI_BASE_ADDRESS_IO_MASK;
		PDEBUG("Function 0 IO Base Address = %#lx\n", ioaddr);

		if (pcidev_joy) {
			ioaddr1 = ioaddr1 & PCI_BASE_ADDRESS_IO_MASK;
			PDEBUG("Function 1 IO Base Address = %#lx\n", ioaddr1);
		}
		
		irq = pcidev->irq;
		
		/*
		 *	Some BIOSes screw this stuff up. Its worth reporting
		 *	to the user
		 */
		 
		if (irq == 0)
		{
			printk(KERN_ERR "sblive: No IRQ has been assigned to this device. Check your BIOS settings.\n");
			continue;
		}
		
		PDEBUG("IRQ Line = %lu\n", irq);

		pci_read_config_byte(pcidev, PCI_REVISION_ID, &ucRevID);
		PDEBUG("Revision ID = %u\n", ucRevID);

		pci_read_config_dword(pcidev, PCI_SUBSYSTEM_VENDOR_ID, &serial);
		PDEBUG("Serial No = %08x\n", serial);
#endif				// LINUX_VERSION_CODE

		// Initialize device struct
		hw_obj = kmalloc(sizeof(*hw_obj), GFP_KERNEL);
		if (!hw_obj) {
			printk(KERN_WARNING "emu10k1: out of memory\n");
			continue;
		}
#ifdef MEMTEST
		PDEBUG("\nmain.c: kmalloc: [%p]\n", hw_obj);
#endif
		memset(hw_obj, 0, sizeof(*hw_obj));

#if LINUX_VERSION_CODE >= 0x020100
		hw_obj->pcidev_joy = pcidev_joy;
		hw_obj->joybase = ioaddr1;
#endif
		hw_obj->bIsAPS = (serial == APS_SERIAL);

		if (hw_obj->bIsAPS)
			printk(KERN_INFO "E-mu APS at %#lx on irq %lu\n",ioaddr,irq);
		else
			printk(KERN_INFO "Creative SBLive! at %#lx on irq %lu\n",ioaddr,irq);

#ifdef MEMTEST
		PDEBUG("\nmain.c: kmalloc: [%p]\n", config);
#endif
		memset(config, 0, sizeof(*config));

		config->vendorid = PCI_VENDOR_ID_CREATIVE;
		config->serialno = serial;
		config->logdevid = PCI_DEVICE_ID_CREATIVE_EMU10K1;
		config->chiprev = ucRevID;

		config->nummemwindows = 0;

		config->numioports = 1;
		config->ioportbase[0] = ioaddr;
		config->ioportlenh[0] = EMU10K1_EXTENT;

		config->numirq = 1;
		config->IRQregs[0] = irq;
		config->irqattr[0] = 0;	// Doesn't matter            

		config->numdma = 0;

		// Reserve I/O region
		if (check_region(ioaddr, EMU10K1_EXTENT)) {
			printk(KERN_ERR "emu10k1: io ports %#lx-%#lx in use\n",
			       ioaddr, ioaddr + EMU10K1_EXTENT - 1);
			goto err_region1;
		}
		request_region(ioaddr, EMU10K1_EXTENT, "emu10k1");

#if LINUX_VERSION_CODE >= 0x020100
		if (check_region(ioaddr1, JOY_EXTENT)) {
			printk(KERN_ERR "emu10k1: io ports %#x-%#x in use\n",
			       (unsigned int) ioaddr1, (unsigned int) ioaddr1 + JOY_EXTENT - 1);
			goto err_region2;
		}
		request_region(ioaddr1, JOY_EXTENT, "emu10k1 joystick");
#endif

		// Register devices
#if LINUX_VERSION_CODE < 0x020100
		hw_obj->audio_num = register_sound_dsp(&emu10k1_audio_fops);
#else
		hw_obj->audio_num = register_sound_dsp(&emu10k1_audio_fops, -1);
#endif
		if (hw_obj->audio_num < 0) {
			printk(KERN_ERR "emu10k1: cannot register audio device!\n");
			goto err_dev1;
		}
		if (!hw_obj->bIsAPS) {
#if LINUX_VERSION_CODE < 0x020100
			hw_obj->mixer_num = register_sound_mixer(&emu10k1_mixer_fops);
#else
			hw_obj->mixer_num = register_sound_mixer(&emu10k1_mixer_fops, -1);
#endif
			if (hw_obj->mixer_num < 0) {
				printk(KERN_ERR "emu10k1: cannot register mixer device!\n");
				goto err_dev2;
			}
		}

#if LINUX_VERSION_CODE < 0x020100
		hw_obj->midi_num = register_sound_midi(&emu10k1_midi_fops);
#else
		hw_obj->midi_num = register_sound_midi(&emu10k1_midi_fops, -1);
#endif
		if (hw_obj->midi_num < 0) {
			printk(KERN_ERR "emu10k1: cannot register MIDI device!\n");
			goto err_dev4;
		}
#if LINUX_VERSION_CODE >= 0x020100
		if (pcidev_joy) {
			if ((joystick[index] & ~0x18) == 0x200) {
				if (check_region(joystick[index], JOY_EXTENT)) {
					printk(KERN_ERR "emu10k1: joystick io ports %lx-%lx in use\n",
						joystick[index], joystick[index] + JOY_EXTENT - 1);
				} else {
					// write joystick base address to PCI configuration space
					pci_write_config_dword(pcidev_joy, PCI_BASE_ADDRESS_0, joystick[index]);
#ifdef EMU10K1_DEBUG
					{
						u32 base;
						pci_read_config_dword(pcidev_joy, PCI_BASE_ADDRESS_0, &base);
						PDEBUG(" read back joystick base address is %x\n", base);
					}
#endif
				}
			}
		}
#endif				// LINUX_VERSION_CODE

		// Register-level initialization goes here...
		PDEBUG("Initializing hardware...\n");
		// IRQ reservation happens in halInit
		if (halInit(hw_obj, config, 0, &hwflags) != CTSTATUS_SUCCESS) {
			printk(KERN_ERR "emu10k1: cannot initialize device!\n");
			goto err_hal_init;
		}
#if LINUX_VERSION_CODE < 0x020301
		hw_obj->open_sem = MUTEX;
#else
		init_MUTEX(&hw_obj->open_sem);
#endif /* LINUX_VERSION_CODE < 0x020301 */
		hw_obj->open_mode = 0;
#if LINUX_VERSION_CODE < 0x020301
		init_waitqueue(&hw_obj->open_wait);
#else
		init_waitqueue_head(&hw_obj->open_wait);
#endif /* LINUX_VERSION_CODE < 0x020301 */

		PDEBUG("Hardware initialized. TRAM allocated: %u bytes\n", (unsigned int) hw_obj->tmemsize);

		// Device-specific initialization goes here    
		if (hw_obj->bIsAPS) {
			audio_init(hw_obj, "E-mu APS");
			sblive_ecard_init(hw_obj);
		} else {
			audio_init(hw_obj, "SBLive!");
			sblive_mixer_init(hw_obj);
		}

		midi_init(hw_obj);

		// AOK, update global device list
		hw_obj->next = sblive_devs;
		sblive_devs = hw_obj;

		hw_obj->prev = NULL;
		if (hw_obj->next)
			hw_obj->next->prev = hw_obj;
		continue;

		// Error handlers
		// TODO: These must reflect the device init order...
err_hal_init:
		halExit(hw_obj);	// Should this be called?

		unregister_sound_midi(hw_obj->midi_num);
err_dev4:
		if (!hw_obj->bIsAPS)
			unregister_sound_mixer(hw_obj->mixer_num);
err_dev2:
		unregister_sound_dsp(hw_obj->audio_num);
err_dev1:
		//free_irq(hw_obj->irq, hw_obj);
		//  err_irq:
#if LINUX_VERSION_CODE >= 0x020100
		release_region(ioaddr1, JOY_EXTENT);
err_region2:
#endif
		release_region(ioaddr, EMU10K1_EXTENT);
err_region1:
		kfree(hw_obj);
#ifdef MEMTEST
		PDEBUG("\nmain.c: kfree: [%p]\n", hw_obj);
#endif
	}

	if (!sblive_devs)
		return -ENODEV;

	sblive_devs->stat_num = register_sound_special(&emu10k1_sndstat_fops, 6);
	if (sblive_devs->stat_num < 0)
		printk(KERN_ERR "emu10k1: cannot register sndstat device!\n");
	else
		sndstat_init();

	PDEBUG("Module successfully initialized!\n");
	return 0;
}

#ifdef MODULE

#if LINUX_VERSION_CODE >= 0x020100
MODULE_PARM(joystick, "1-" __MODULE_STRING(MAX_EMU10K1) "i");
MODULE_PARM_DESC(joystick, "sets address and enables joystick interface (still need separate driver)");
MODULE_AUTHOR("Bertrand Lee, Cai Ying. (Email to: linux_bug@soundblaster.com)");
MODULE_DESCRIPTION("Creative EMU10K1 Driver v0.4\nCopyright (C) 1999 Creative Technology Ltd.");
#endif				// LINUX_VERSION_CODE

// Driver exit routine
void cleanup_module(void)
{
	struct sblive_hw *hw_obj;

	PDEBUG("cleanup_module() called\n");

	if (sblive_devs && sblive_devs->stat_num >= 0)
		unregister_sound_special(6);

	while ((hw_obj = sblive_devs)) {
		sblive_devs = sblive_devs->next;

#if LINUX_VERSION_CODE >= 0x020100
		// write original joystick base address to PCI configuration space
		pci_write_config_dword(hw_obj->pcidev_joy, PCI_BASE_ADDRESS_0, hw_obj->joybase);
#ifdef EMU10K1_DEBUG
		{
			u32 base;
			pci_read_config_dword(hw_obj->pcidev_joy, PCI_BASE_ADDRESS_0, &base);
			PDEBUG("cleanup_module: read back joystick base address is %x\n", base);
		}
#endif
#endif

		audio_exit(hw_obj);
		midi_exit(hw_obj);

		/* IRQ is freed in halExit */
		halExit(hw_obj);
		PDEBUG("halExit completed\n");

		release_region(hw_obj->hwaddr, EMU10K1_EXTENT);
#if LINUX_VERSION_CODE >= 0x020100
		release_region(hw_obj->joybase, JOY_EXTENT);
#endif
		unregister_sound_dsp(hw_obj->audio_num);
		if (!hw_obj->bIsAPS)
			unregister_sound_mixer(hw_obj->mixer_num);
		unregister_sound_midi(hw_obj->midi_num);
		kfree(hw_obj);
#ifdef MEMTEST
		PDEBUG("\nmain.c: kfree: [%p]\n", hw_obj);
#endif
	}

	//kfree(g_config);
#ifdef MEMTEST
	//PDEBUG("\nmain.c: kfree: [%p]\n", g_config);
#endif
	PDEBUG("module unloaded\n");
}

#endif				// MODULE
