/*================================================================
 * preload samples in a midi file using AWElib
 *
 * Copyright (C) 1996-1998 Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/wait.h>
#include "config.h"
#include "util.h"
#include "midievent.h"
#include "controls.h"
#include "seq.h"
#include "channel.h"

#include <awebank.h>
#include <sfopts.h>

typedef struct _SubRec {
	char *args;
	struct _SubRec *next;
} SubRec;

static int nsubs;
static SubRec *psub;

static LoadList *pload;
static char *fontfile;

static int load_all_files(MidiInfo *mp);
static int load_fonts(MidiInfo *mp, char *file, LoadList *lp, int bank, int load_alt);

/* clear variables */
void init_preload(MidiInfo *mp)
{
	nsubs = 0;
	psub = NULL;

	pload = NULL;
	fontfile = NULL;
}

/* load fonts */
void load_partial_fonts(MidiInfo *mp)
{
	int rc = 1;
	SubRec *q, *nextq;

	if (!nsubs && !pload) {
		if (fontfile) free(fontfile);
		return;
	}

	if (mp->incremental)
		rc = load_all_files(mp);

	if (rc == AWE_RET_NOMEM) {
		seq_remove_samples();
		if (load_all_files(mp)) {
			if (ctl)
				ctl->cmsg(CMSG_INFO, -1, "preload: couldn't load all fonts");
		}
	}

	if (ctl) {
		ctl->cmsg(CMSG_INFO, 1, "preload: rest of memory = %d kB", seq_mem_avail() / 1024);
	}

	/* free sub list */
	for (q = psub; q; q = nextq) {
		nextq = q->next;
		free(q->args);
		free(q);
	}

	/* free part list */
	if (pload)
		free_loadlist(pload);

	/* free font file name */
	if (fontfile)
		safe_free(fontfile);
}

/* load all sub fonts and partial fonts */
static int load_all_files(MidiInfo *mp)
{
	SubRec *p;
	int rc;

	/* load sub fonts */
	for (p = psub; p; p = p->next) {
		if ((rc = load_fonts(mp, p->args, NULL, 1, FALSE)) != 0)
			return rc;
	}
	if (pload)
		return load_fonts(mp, fontfile, pload, 0, TRUE);
	return 0;
}

/* load one font file */
static int load_fonts(MidiInfo *mp, char *file, LoadList *lp, int bank, int load_alt)
{
	awe_read_option_file(file);
	awe_option.default_bank = bank;
	return awe_load_bank(file, lp, FALSE);
}

/* parse MIDI file and retrieve required instruments */
void preload_sample(MidiInfo *mp, char *file)
{
	MidiEvent *ev;
	int i;

	if (fontfile)
		free(fontfile);
	fontfile = safe_strdup(file);

	channel_init(mp);

	ev = mp->list;
	for (i = 0; i < mp->nlists; i++, ev++) {
		do_midi_event(ev, mp, EV_PRELOAD);
	}
}

/* add the intstrument to the list */
void add_preload(int chn, int preset, int bank, int keynote)
{
	SFPatchRec pat;
	LoadList *rec;

	pat.preset = preset;
	pat.bank = bank;
	pat.keynote = keynote;
	for (rec = pload; rec; rec = rec->next) {
		if (awe_match_preset(&rec->pat, &pat))
			return;
	}
	pload = add_loadlist(pload, &pat, NULL);
}

/* add the font to sub list */
void load_sub_sf(MidiInfo *mp, char *args)
{
	SubRec *p;

	p = (SubRec*)safe_malloc(sizeof(SubRec));
	p->args = safe_strdup(args);
	p->next = psub;
	psub = p;
	nsubs++;
}

