/*================================================================
 * miditext:
 *	visualize MIDI event sequences to stdout
 *
 * usage: miditext [-options] midifile
 *
 * Copyright (C) 1996-1998 Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *================================================================*/

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "config.h"
#include "midievent.h"
#include "util.h"
#include "controls.h"
#include "awe_effect.h"
#include "channel.h"
#include "options.h"

static int cmsg(int type, int verbosity_level, char *fmt, ...);

static ControlMode dummy_ctl = 
{
  "dumb interface", 'd',
  FALSE,
  FALSE,
  FALSE,
  0,0,0,0,
  NULL,NULL,NULL,NULL,NULL,cmsg,
  NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,
};

ControlMode *ctl = &dummy_ctl;

MidiInfo glinfo;
int verbose = 0, debug = 0;
int seqfd, awe_dev;
#define OPTION_FLAGS	"hvG:T:"
#define OPT_CHANNEL	OPT_LOCAL
static awe_option_args long_options[] = {
	{"help", 1, 0, OPT_HELP},
	{"verbose", 2, 0, OPT_VERBOSE},
	{"gsmacro", 2, 0, OPT_GSMACRO},
	{"tracks", 1, 0, OPT_TRACKS},
	{"multipart", 1, 0, OPT_MULTIPART},
	{"channel", 1, 0, OPT_CHANNEL},
	{0, 0, 0, 0},
};
static int option_index;


static int cmsg(int type, int verbosity_level, char *fmt, ...)
{
	va_list ap;
	if (ctl->verbosity <= verbosity_level) return 0;
	va_start(ap, fmt);
	vprintf(fmt, ap);
	printf("\n");
	va_end(ap);
	return 0;
}

static struct control_label {
	int type;
	char *str;
} labels[] = {
	{CTL_MODWHEEL, "modwheel"},
	{CTL_MAIN_VOLUME, "main_volume"},
	{CTL_PAN, "pan"},
	{CTL_EXPRESSION, "expression"},
	{CTL_SUSTAIN, "sustain"},
	{CTL_SOSTENUTO, "sostenuto"},
	{CTL_CHORUS_DEPTH, "chorus"},
	{CTL_REVERB_DEPTH, "reverb"},
}, names[] = {
	{CTL_SOFT_PEDAL, "soft_pedal"},
	{CTL_PORTAMENTO, "portament"},
	{70, "xg_variation"},
	{71, "xg_cutoff"},
	{72, "xg_release"},
	{73, "xg_attack"},
	{74, "xg_brightness"},
};

static void PrintControl(MidiEvent *e)
{
	int i;
	for (i = 0; i < numberof(labels); i++) {
		if (labels[i].type == CtrlOf(e)) {
			printf("%-16s (%02d) %d %d\n", labels[i].str,
			       e->channel, CtrlOf(e), CValOf(e));
			return;
		}
	}
	for (i = 0; i < numberof(names); i++) {
		if (names[i].type == CtrlOf(e)) {
			printf("%-16s (%02d) %d %d %s\n", "control",
			       e->channel, CtrlOf(e), CValOf(e),
			       names[i].str);
			return;
		}
	}
	printf("%-16s (%02d) %d %d\n", "control",
	       e->channel, CtrlOf(e), CValOf(e));
}


static void PrintCD(char *msg, MidiEvent *e)
{
	printf("%-16s (%02d) %d %d\n", msg, e->channel, KeyOf(e), VelOf(e));
}

static void PrintCB(char *msg, MidiEvent *e)
{
	printf("%-16s (%02d) %d\n", msg, e->channel, e->p.val);
}

static void NRPNEvent(MidiEvent *ev)
{
	int type, val, cval;

	type = ev->p.par[0];
	val = ev->p.par[1] - 8192;
	if (type < 0 || type >= num_awe_effects)
		return;

	cval = awe_effects[type].convert(val);
	printf("%-16s (%02d) %d %d --> %x\n",
	       awe_effects[type].name, ev->channel, type, val,
	       (unsigned short)cval);
}

void print_usage(void)
{
	fprintf(stderr, "usage: miditext [-options] midifile\n");
	print_general_options(stderr, OPTION_FLAGS, long_options);
	fprintf(stderr, "  --channel=val: output only the specified channel\n");
}

static int ChannelEvents(int type)
{
	return (type == ME_NOTEON || type == ME_NOTEOFF ||
		type == ME_KEYPRESSURE || type == ME_CHNPRESSURE ||
		type == ME_PITCH_SENS || type == ME_PITCHWHEEL ||
		type == ME_CONTROL || type == ME_PROGRAM ||
		type == ME_TONE_BANK || type == ME_FINETUNE ||
		type == ME_COARSETUNE || type == ME_AWE_FX ||
		type == ME_GS_FX);
}

static int output_flags;

int main(int argc, char **argv)
{
	FILE *fp;
	MidiEvent *ev;
	int piped;
	int i, cp;

	output_flags = 0;
	memcpy(&glinfo, &gl_default, sizeof(glinfo));
	glinfo.chorus = glinfo.reverb = -1; /* uninitialized */
	while ((cp = awe_getopt(argc, argv, OPTION_FLAGS, long_options, &option_index)) != -1) {
		switch (cp) {
		case OPT_CHANNEL:
			output_flags |= DRUMBIT(atoi(optarg) - 1);
			break;
		default:
			general_options(OPTION_FLAGS, cp);
		}
	}
	ctl->verbosity = verbose;
	if (optind >= argc) {
		print_usage();
		exit(1);
	}

	if ((fp = CmpOpenFile(argv[optind], &piped)) == NULL) {
		fprintf(stderr, "can't open midi file %s\n", argv[optind]);
		exit(1);
	}

	ev = ReadMidiFile(fp, &glinfo);
	if (ev == NULL)
		exit(0);

	printf("#midievents %d\n", glinfo.nlists);
	printf("#drumflag 0x%x\n", glinfo.drumflag);
	printf("#midimode %d\n", glinfo.midi_mode);
	printf("#multipart %d\n", glinfo.multi_part);
	if (glinfo.chorus >= 0)
		printf("#chorus %d\n", glinfo.chorus);
	if (glinfo.reverb >= 0)
		printf("#reverb %d\n", glinfo.reverb);
	if (glinfo.nlists > 0)
		printf("#totaltime %d\n", ev[glinfo.nlists-1].csec);
	else
		printf("#totaltime 0\n");
	for (i = 0; i < glinfo.nlists; i++) {
		if (ChannelEvents(ev[i].type)) {
			if (output_flags &&
			    !(output_flags & DRUMBIT(ev[i].channel)))
				continue;
		}
		printf("%-15d %-15d ", ev[i].csec, ev[i].time);
		switch (ev[i].type) {
		case ME_NOTEON:
			PrintCD("note_on", ev + i); break;
		case ME_NOTEOFF:
			PrintCD("note_off", ev + i); break;
		case ME_KEYPRESSURE:
			PrintCD("key_press", ev + i); break;
		case ME_CHNPRESSURE:
			PrintCB("chn_press", ev + i); break;
		case ME_PITCH_SENS:
			PrintCB("pitch_sens", ev + i); break;
		case ME_PITCHWHEEL:
			PrintCB("pitch_wheel", ev + i); break;
		case ME_CONTROL:
			PrintControl(ev + i); break;
		case ME_PROGRAM:
			PrintCB("program", ev + i); break;
		case ME_RESET_CONTROLLERS:
			PrintCB("reset_control", ev + i); break;
		case ME_ALL_NOTES_OFF:
			printf("all_notes_off \n"); break;
		case ME_ALL_SOUNDS_OFF:
			printf("all_sounds_off \n"); break;
		case ME_TONE_BANK:
			PrintCD("tone_bank", ev + i); break;
		case ME_TEMPO:
			printf("%-16s %d\n", "tempo", ev->p.val);
			break;
		case ME_LYRIC:
			printf("%-16s\n", "lyric");
			break;
		case ME_MASTER_VOLUME:
			PrintCB("master_volume", ev + i);
			break;
		case ME_AWE_FX:
			NRPNEvent(ev + i);
			break;
		case ME_GS_FX:
			printf("%-16s (%02d) %d %d", "gs_nrpn",
			       ev[i].channel, ev[i].p.par[0], ev[i].p.par[1]);
			{int p;
			for (p = 0; p < num_gs_effects; p++) {
				if (gs_effects[p].control == ev[i].p.par[0]) {
					printf(" %s", gs_effects[p].name);
					break;
				}
			}
			}
			printf("\n");
			/*PrintCD("gs_nrpn", ev + i);*/
			break;
		case ME_FINETUNE:
			PrintCB("fine_tune", ev + i);
			break;
		case ME_COARSETUNE:
			PrintCB("coarse_tune", ev + i);
			break;
		case ME_EOT:
			printf("end_of_tune \n");
			exit(0);
		}
	}
	return 0;
}
