/*
  **********************************************************************
  *
  *     Copyright 1999, 2000 Creative Labs, Inc.
  *
  **********************************************************************
  *
  *     Date                 Author               Summary of changes
  *     ----                 ------               ------------------
  *   
  *     October 20, 1999     Andrew de Quincey    Rewrote and extended
  *                          Lucien Murray-Pitts  original incomplete 
  *                                               driver.
  *
  *     April 18, 1999       Andrew Veliath       Original Driver
  *                                               implementation
  *
  **********************************************************************
  *
  *     This program is free software; you can redistribute it and/or
  *     modify it under the terms of the GNU General Public License as
  *     published by the Free Software Foundation; either version 2 of
  *     the License, or (at your option) any later version.
  *
  *     This program is distributed in the hope that it will be useful,
  *     but WITHOUT ANY WARRANTY; without even the implied warranty of
  *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  *     GNU General Public License for more details.
  *
  *     You should have received a copy of the GNU General Public
  *     License along with this program; if not, write to the Free
  *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
  *     USA.
  *
  **********************************************************************
  */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include "player.h"
#include "config.h"
#include "interface.h"
#include "vobtitle.h"

/* UNCOMMENT THIS TO TRY OVERLAY IN A WINDOW */
//#include "overlay.h"

void initialize(int argc, char** argv, player_info_t* player_info, dxr2_status_info_t* dxr2_info)
{
  memset(player_info, 0, sizeof(player_info_t));
  player_info->mpegFD=player_info->driveFD=player_info->dxr2FD=-1;
  memset(dxr2_info, 0, sizeof(dxr2_status_info_t));

  init_dxr2_info(dxr2_info);
  strcpy(player_info->uCode_file, "dvd1.ux");

  read_config(player_info, dxr2_info);
  if(!init_and_parse_args(argc, argv, player_info, dxr2_info)) {
    cleanup_and_exit(player_info);
  }

  if (player_info->file_name[0]==0 || argc < 2) {
    destroy_interface();
    printf("Syntax: %s [-cTO] [-s <bitstream type>] [-a <audio stream>]\n",argv[0]);
    printf("                  [-f <audio freq>] [-W <TV type>] [-u <audio type>]\n");
    printf("                  [-w <audio width>] [-r <output ratio>] [-t <output type>]\n");
    printf("                  [-S <subtitle jid>] [-g <geom>] [-p <crop>] [-b <brightness>]\n");
    printf("                  [-A <angle>] <filename>\n");
    printf("\n");
    printf("-c turns on CSS decryption\n");
    printf("-a will select the audio stream to decode (from 0 to 7)\n");
    printf("-r is either 0 1 or 2, 0 being letterbox, 1 being 'normal' and 2 being pan/scan\n");
    printf("-t is either 0 to 7 corresponding to: NTSC,NTSC_60,PAL_M,PAL_M_60,PAL_BDGHI,PAL_N,PAL_Nc,PAL_60\n");
    printf("-q selects the audio frequency.  0-4 corresponding to:  44.1, 48. 96, 2205, 32\n");
    printf("-W Selects whether you have a Widescreen TV(1) or a Normal Tv(0)\n");
    printf("-w selects the audio width.  0-2 corresponding to: 16, 20, 24\n");
    printf("-s selects the bitstream type.  0-4 corresponding to: VOB(DVD), CDROM VCD, MPEG VCD, CDDA, Unknown\n");
    printf("-u selects the audio bitstream type.  0-2 corresponding to: AC3 / MPEG / LPCM\n");
    printf("-S select subtitle language. 0-<number of subtitles languages on the DVD -1>\n");
    printf("-T turns on IFO parsing.  NOTE: you _MUST_ specify the *_01.VOB file for this to work\n");
    printf("-A selects which angle to view.  0-<maxangle>\n");
    printf("-O turns on the VGA overlay\n");
    printf("-g sets the geometry of the VGA overlay.  standard wxh+x+y format.\n");
    printf("-p sets the cropping info.  <left shift>x<right shift>x<top shift>x<bottom shift>\n");
    printf("-b sets the brightness for the VGA overlay.  0-63\n");
    exit(1);
  }

  open_files(player_info);

  install_firmware(player_info);

  // reset the player
  dxr2_init(player_info->dxr2FD);

  // do CSS auth (if dvd device is supplied)  
  if (player_info->encrypted) 
    authenticate_disc(player_info);  

  if (player_info->title)
    vob_choose_angle(player_info->angle);

  // setup the rest of the parameters
  dxr2_set_params(dxr2_info);



  // **************************************************
  // THIS SHOULD BE CHANGED. IT IS A NASTY HACK!
  // initialise the overlay if necessary
  if(dxr2_info->overlay_mode.arg != DXR2_OVERLAY_DISABLED) {

    /*
    dxr2_vgaParams_t vgaBuf;
    int pid;
    int xRes;
    int yRes;
    dxr2_sixArg_t buf6;
    dxr2_oneArg_t buf1;
    dxr2_twoArg_t buf2;
    int dxr2FD;
    
    dxr2FD = player_info->dxr2FD;


    // setup for measuring overlay
    getScreenRes(&xRes, &yRes);
    
    // OK, fork off another process to do the whitescreen thang.
    pid = fork();
    if (pid == 0) { // i.e. the child process
      
      whitescreen();
      exit(0);
    }

    // set colour key
    buf6.arg1 = 0x80;
    buf6.arg2 = 0xff;
    buf6.arg3 = 0x80;
    buf6.arg4 = 0xff;
    buf6.arg5 = 0x80;
    buf6.arg6 = 0xff;
    ioctl(dxr2FD, DXR2_IOC_SET_OVERLAY_COLOUR, &buf6);

    // set ratio to 1:1
    buf1.arg = 1000;
    ioctl(dxr2FD, DXR2_IOC_SET_OVERLAY_RATIO, &buf1);

    // set overlay position to detection position
    buf2.arg1 = 100;
    buf2.arg2 = 3;
    ioctl(dxr2FD, DXR2_IOC_SET_OVERLAY_POSITION, &buf2);

    // make the thing the width/height of the screen
    buf2.arg1 = xRes;
    buf2.arg2 = yRes;
    ioctl(dxr2FD, DXR2_IOC_SET_OVERLAY_DIMENSION, &buf2);

    // turn window AND colour keying on
    buf1.arg = DXR2_OVERLAY_WINDOW_COLOUR_KEY;
    ioctl(dxr2FD, DXR2_IOC_SET_OVERLAY_MODE, &buf1);

    // calculate VGA parameters
    vgaBuf.xScreen = xRes;
    vgaBuf.yScreen = yRes;
    vgaBuf.hOffWinKey = 100;
    vgaBuf.vOffWinKey = 3;
    ioctl(dxr2FD, DXR2_IOC_CALCULATE_VGA_PARAMETERS, &vgaBuf);

    // OK, kill off the whitescreen process
    kill(pid, 9);

    // OK, setup the overlay properly
    dxr2_setup_overlay(&vgaBuf, dxr2_info);

    */

    player_info->do_overlay = 1;
  }
  // **************************************************

  /* UNCOMMENT THIS TO TRY OVERLAY IN A WIN */
  //init_overlay(argc, argv, dxr2_info);

}

void init_dxr2_info(dxr2_status_info_t* dxr2_info)
{
  // Default values
  dxr2_info->tv_format.arg= DXR2_OUTPUTFORMAT_NTSC;
  // Set this to an invalid value so that we set it later based on tv_format
  dxr2_info->video_freq.arg1=-1;
  dxr2_info->video_freq.arg2=0x2d0;
  dxr2_info->video_freq.arg3=0x1e0;
  dxr2_info->output_aspect_ratio.arg=DXR2_ASPECTRATIO_4_3;
  dxr2_info->source_aspect_ratio.arg=DXR2_ASPECTRATIO_4_3;
  dxr2_info->scaling_mode.arg=DXR2_ASPECTRATIOMODE_LETTERBOX;
  dxr2_info->bitstream_type.arg=DXR2_BITSTREAM_TYPE_MPEG_VOB;
  dxr2_info->macro_vision.arg=DXR2_MACROVISION_OFF;
  dxr2_info->pixel_mode.arg=DXR2_PIXEL_CCIR601;
  dxr2_info->interlaced_mode.arg=DXR2_INTERLACED_ON;
  dxr2_info->x75ire_mode.arg=DXR2_75IRE_OFF;
  dxr2_info->volume.arg=19;
  dxr2_info->mute.arg=DXR2_AUDIO_MUTE_OFF;
  dxr2_info->audio_width.arg=DXR2_AUDIO_WIDTH_16;
  dxr2_info->audio_freq.arg=DXR2_AUDIO_FREQ_441;
  dxr2_info->iec_output_mode.arg=DXR2_IEC958_ENCODED;
  dxr2_info->subpicture.arg=DXR2_SUBPICTURE_OFF;
  dxr2_info->subpicture_stream_id.arg1=DXR2_STREAM_SUBPICTURE;
  dxr2_info->subpicture_stream_id.arg2=0;
  dxr2_info->audio_stream.arg1=DXR2_STREAM_AUDIO_AC3;
  dxr2_info->audio_stream.arg2=0;
  dxr2_info->video_stream.arg1=DXR2_STREAM_VIDEO;
  dxr2_info->video_stream.arg2=0;

  /* VGA OVERLAY INFO */
  dxr2_info->overlay_crop.arg1=0; // left
  dxr2_info->overlay_crop.arg2=0; // right
  dxr2_info->overlay_crop.arg3=20; // top
  dxr2_info->overlay_crop.arg4=20;  // bottom
  dxr2_info->overlay_pos.arg1=81; // origin X
  dxr2_info->overlay_pos.arg2=36; // origin Y
  dxr2_info->overlay_dim.arg1=800; // size horiz
  dxr2_info->overlay_dim.arg2=600; // size vert
  dxr2_info->overlay_mode.arg=DXR2_OVERLAY_DISABLED;
  dxr2_info->picture.arg1 = 63; // common gain (0-63)
  dxr2_info->picture.arg2 = 10; // red-gain (0-15)
  dxr2_info->picture.arg3 = 0; // green-gain (0-15)
  dxr2_info->picture.arg4 = 10; // blue-gain (0-15)
}

int init_and_parse_args(int argc, char** argv, player_info_t* player_info, dxr2_status_info_t* dxr2_info)
{ 
  char optionsString[] = "f:ca:g:p:r:t:q:w:s:u:v:OTS:W:A:b:"; 
  char geom[40];
  char crop[40];
  int ratio, output_type=-1, current_opt, audio_freq, audio_width;
  int video_freq=-1;
  int audio_type=-1;
  int subtitle_id=-1;
  
  strcpy(player_info->file_name, *(argv+argc-1));
  strcpy(geom, "");
  strcpy(crop, "");

  while((current_opt = getopt(argc,argv,optionsString))!=-1) {
    if(current_opt == 'a')
      sscanf(optarg,"%d",&dxr2_info->audio_stream.arg2);
    else if (current_opt=='r')
      sscanf(optarg,"%d",&ratio);
    else if (current_opt=='t')
      sscanf(optarg,"%d",&output_type);
    else if (current_opt=='f')
      sscanf(optarg,"%s",player_info->file_name);
    else if(current_opt =='c')
      player_info->encrypted = 1;
    else if(current_opt == 'g')
      sscanf(optarg,"%s", geom);
    else if(current_opt == 'p')
      sscanf(optarg, "%s", crop);
    else if(current_opt =='w')
      sscanf(optarg,"%d",&dxr2_info->audio_width.arg);
    else if(current_opt =='q')
      sscanf(optarg,"%d",&dxr2_info->audio_freq.arg);
    else if(current_opt =='s')
      sscanf(optarg,"%d",&dxr2_info->bitstream_type.arg);
    else if(current_opt =='u')
      sscanf(optarg,"%d",&audio_type);
    else if(current_opt == 'v')
      sscanf(optarg, "%d", &video_freq);
    else if(current_opt == 'A')
      sscanf(optarg, "%d", &player_info->angle);
    else if(current_opt == 'T')
      player_info->title = 1;
    else if(current_opt == 'W')
      sscanf(optarg, "%d", &dxr2_info->output_aspect_ratio.arg);
    else if(current_opt =='S') {
      dxr2_info->subpicture.arg=DXR2_SUBPICTURE_ON;
      sscanf(optarg,"%d",&subtitle_id);
    }
    else if(current_opt=='O') 
      dxr2_info->overlay_mode.arg=DXR2_OVERLAY_WINDOW_KEY;
    else if(current_opt=='b')
      sscanf(optarg, "%d", &dxr2_info->picture.arg1);
  }

  if(subtitle_id >= 0)
    dxr2_info->subpicture_stream_id.arg2=subtitle_id;

  if (ratio == 0) 
    dxr2_info->scaling_mode.arg = DXR2_ASPECTRATIOMODE_LETTERBOX;
  else if (ratio == 2)
    dxr2_info->scaling_mode.arg = DXR2_ASPECTRATIOMODE_PAN_SCAN;
  else if (ratio == 2)
    dxr2_info->scaling_mode.arg = DXR2_ASPECTRATIOMODE_NORMAL;

  if (output_type >=0  && output_type <= 7) {
    dxr2_info->tv_format.arg = output_type;
  }
  else if(output_type != -1) {
    print_error("Invalid Output format.  must be 0-6 corresponding to: \n");
    print_error("NTSC, NTSC_60, PAL_M, PAL_M_60, PAL_BDGHI, PAL_N, PAL_Nc\n");
  }

  if(dxr2_info->audio_freq.arg < 0 || dxr2_info->audio_freq.arg > 4) {
    print_error("Invalid frequency.  0=44.1 KHz, 1=48 KHz, 2=96 KHz,\n");
    print_error("   3=2205 Hz, 4=32 Khz.\n");
    return 0;;
  }
 
  if(dxr2_info->audio_width.arg < 0 || dxr2_info->audio_width.arg > 2) {
    print_error("Invalid width.  0=16, 1=20, 2=24\n");
    return 0;
  }

  if(dxr2_info->bitstream_type.arg < 0 || dxr2_info->bitstream_type.arg > 4) {
    print_error("Invalid biststream type.  0=VOB (DVD), 1=CDROM VCD (raw sector mode?), 2=MPEGVCD, 3=CDDA, 4=unknown\n");
    return 0;
  }

  // setup audio types properly (this should probably be configurable?)
  if(audio_type == -1 && dxr2_info->audio_stream.arg1==0) {
    // if we havn't specified something, either here or in the config file...
    if(dxr2_info->bitstream_type.arg == DXR2_BITSTREAM_TYPE_MPEG_VOB) {
      dxr2_info->audio_stream.arg1 = DXR2_STREAM_AUDIO_AC3;
    }
    else if (dxr2_info->bitstream_type.arg == DXR2_BITSTREAM_TYPE_MPEG_VCD) {
      dxr2_info->audio_stream.arg1 = DXR2_STREAM_AUDIO_MPEG;
    }
  }
  else if(audio_type == -1)
    ; // specified in the config file, but not at command line.  do nothing.
  else if(audio_type < 0 || audio_type > 2) {
    print_error("Invalid audio type.  0==AC3, 1==MPEG, 2==LPCM\n");
    return 0;
  }
  else
    dxr2_info->audio_stream.arg1 = audio_type+2;  // shift to correspond to correct numbers.

  // If we have not set the video frequency yet, set it to default.
  if(dxr2_info->video_freq.arg1==-1 && video_freq == -1)
    if ((output_type == DXR2_OUTPUTFORMAT_NTSC) ||
	(output_type == DXR2_OUTPUTFORMAT_NTSC_60)) {
      
      video_freq = DXR2_SRC_VIDEO_FREQ_30;
    }
    else {

      video_freq = DXR2_SRC_VIDEO_FREQ_25;
    }

  if(player_info->angle < 0) {
    print_error("ERROR: -A must choose an angle > 0\n");
    destroy_interface();
    exit(1);
  }

  if(video_freq==0) 
    dxr2_info->video_freq.arg1 = DXR2_SRC_VIDEO_FREQ_30;
  else if(video_freq==1) 
    dxr2_info->video_freq.arg1 = DXR2_SRC_VIDEO_FREQ_25;

  if(dxr2_info->output_aspect_ratio.arg < 0 || dxr2_info->output_aspect_ratio.arg > 1) {
    print_error("ERROR: -W must have either a 0 or a 1 as its argument.\n");
    destroy_interface();
    exit(0);
  }

  if(strcmp(geom, "") != 0)
    parse_geom(dxr2_info, geom);

  if(strcmp(crop, "") != 0)
    parse_crop(dxr2_info, crop);

  if(dxr2_info->picture.arg1 < 0 || dxr2_info->picture.arg1 > 63) {
    print_error("ERROR: invalid brightness level. must be 0-63\n");
    destroy_interface();
    exit(1);
  }

  return 1;
}

void read_config(player_info_t* player_info, dxr2_status_info_t* dxr2_info)
{
  char* config_file="/etc/dxr2player.conf";
  FILE* config;
  char input[20];

  config = fopen(config_file, "r");

  if(errno) {
    destroy_interface();
    fprintf(stderr, "Error loading config file!\n");
    exit(1);
  }
  else {
    while(!feof(config)) {
      fscanf(config, "%s", input);
      if(input[0]=='#') /* skip to end of line */
	while(input[0]!='\n')
	  fscanf(config, "%c", input);
      else {
	if(strcmp(input, "drive:")==0) 
	  fscanf(config, "%s", player_info->dvd_device);
	else if(strcmp(input, "dxr2:")==0) 
	  fscanf(config, "%s", player_info->dxr2_device);
	else if(strcmp(input, "firmware:")==0)
	  fscanf(config, "%s", player_info->uCode_file);
	else if(strcmp(input, "audio_format:")==0) {
	  fscanf(config, "%s", input);
	  if(strcmp(input, "AC3")==0)
	    dxr2_info->audio_stream.arg1=DXR2_STREAM_AUDIO_AC3;
	  else if(strcmp(input, "MPEG")==0)
	    dxr2_info->audio_stream.arg1=DXR2_STREAM_AUDIO_MPEG;
	  else if(strcmp(input, "PCM")==0 || strcmp(input,"LPCM")==0)
	    dxr2_info->audio_stream.arg1=DXR2_STREAM_AUDIO_LPCM;
	  else {
	    destroy_interface();
	    print_error("ERROR in config file %s, argument %s is an invalid stream type!", 
		    config_file, input);
	    exit(1);
	  }
	}
	else if(strcmp(input, "audio_freq:")==0) {
	  fscanf(config, "%s", input);
	  if(strcmp(input, "44.1")==0)
	    dxr2_info->audio_freq.arg=DXR2_AUDIO_FREQ_441;
	  else if(strcmp(input, "48")==0)
	    dxr2_info->audio_freq.arg=DXR2_AUDIO_FREQ_48;
	  else if(strcmp(input, "96")==0)
	    dxr2_info->audio_freq.arg=DXR2_AUDIO_FREQ_96;
	  else if(strcmp(input, "2205")==0)
	    dxr2_info->audio_freq.arg=DXR2_AUDIO_FREQ_2205;
	  else if(strcmp(input, "32")==0)
	    dxr2_info->audio_freq.arg=DXR2_AUDIO_FREQ_32;
	  else {
	    destroy_interface();
	    print_error("ERROR in config file %s, argument %s is an invalid frequency!", 
		    config_file, input);
	    exit(1);
	  }
	}
	else if(strcmp(input, "audio_width:")==0) {
	  fscanf(config, "%s", input);
	  if(strcmp(input, "16")==0)
	    dxr2_info->audio_width.arg=DXR2_AUDIO_WIDTH_16;
	  else if(strcmp(input, "20")==0)
	    dxr2_info->audio_width.arg=DXR2_AUDIO_WIDTH_20;
	  else if(strcmp(input, "24")==0)
	    dxr2_info->audio_width.arg=DXR2_AUDIO_WIDTH_24;
	  else {
	    destroy_interface();
	    print_error("ERROR in config file %s, argument is an %s invalid bitstream width!", 
		    config_file, input);
	    exit(1);
	  }
	}
	else if(strcmp(input, "tv_format:")==0) {
	  fscanf(config, "%s", input);
	  if(strcmp(input, "NTSC")==0)
	    dxr2_info->tv_format.arg=DXR2_OUTPUTFORMAT_NTSC;
	  else if(strcmp(input, "NTSC60")==0)
	    dxr2_info->tv_format.arg=DXR2_OUTPUTFORMAT_NTSC_60;
	  else if(strcmp(input, "PAL_M")==0)
	    dxr2_info->tv_format.arg=DXR2_OUTPUTFORMAT_PAL_M;
	  else if(strcmp(input, "PAL_M_60")==0)
	    dxr2_info->tv_format.arg=DXR2_OUTPUTFORMAT_PAL_M_60;
	  else if(strcmp(input, "PAL_BDGHI")==0)
	    dxr2_info->tv_format.arg=DXR2_OUTPUTFORMAT_PAL_BDGHI;
	  else if(strcmp(input, "PAL_N")==0)
	    dxr2_info->tv_format.arg=DXR2_OUTPUTFORMAT_PAL_N;
	  else if(strcmp(input, "PAL_Nc")==0)
	    dxr2_info->tv_format.arg=DXR2_OUTPUTFORMAT_PAL_Nc;
	  else if(strcmp(input, "PAL_60")==0)
	    dxr2_info->tv_format.arg=DXR2_OUTPUTFORMAT_PAL_60;
	  else {
	    destroy_interface();
	    print_error("ERROR in config file %s, argument is an %s invalid tv format!", 
		    config_file, input);
	    exit(1);
	  }
	}
        else if(strcmp(input, "video_freq:")==0) {
          fscanf(config, "%s", input);
          if(strcmp(input, "30")==0)
            dxr2_info->video_freq.arg1=DXR2_SRC_VIDEO_FREQ_30;
          else if(strcmp(input, "25")==0)
            dxr2_info->video_freq.arg1=DXR2_SRC_VIDEO_FREQ_25;
          else {
	    destroy_interface();
            print_error("ERROR in config file %s, argument %s is an invalid video frequency", config_file, input);
            exit(1);
          }
        }
	else if(strcmp(input, "video_format:")==0) {
	  fscanf(config, "%s", input);
	  if(strcmp(input, "letterbox")==0)
	    dxr2_info->scaling_mode.arg=DXR2_ASPECTRATIOMODE_LETTERBOX;
	  else if(strcmp(input, "normal")==0)
	    dxr2_info->scaling_mode.arg=DXR2_ASPECTRATIOMODE_NORMAL;
	  else if(strcmp(input, "pan_scan")==0)
	    dxr2_info->scaling_mode.arg=DXR2_ASPECTRATIOMODE_PAN_SCAN;
	  else {
	    destroy_interface();
	    print_error("ERROR in config file %s, argument is an %s invalid video format!", 
		    config_file, input);
	    exit(1);
	  }
	}
	else if(strcmp(input, "subpicture:")==0) {
	  fscanf(config, "%s", input);
	  if(strcmp(input, "OFF")==0)
	    dxr2_info->subpicture.arg=DXR2_SUBPICTURE_OFF;
	  else if(strcmp(input, "ON")==0)
	    dxr2_info->subpicture.arg=DXR2_SUBPICTURE_ON;
	  else {
	    destroy_interface();
	    print_error("ERROR in config file %s, argument %s is invalid!", config_file, input);
	    exit(1);
	  }
	}
	else if(strcmp(input, "overlay:")==0) {
	  fscanf(config, "%s", input);
	  if(strcmp(input, "ON")==0)
	    dxr2_info->overlay_mode.arg=DXR2_OVERLAY_WINDOW_KEY;
	  else if(strcmp(input, "OFF")==0)
	    dxr2_info->overlay_mode.arg=DXR2_OVERLAY_DISABLED;
	  else {
	    destroy_interface();
	    print_error("ERROR in config file %s, argument %s is invalid!", config_file, input);
	    exit(1);
	  }
	}
	else if(strcmp(input, "overlay_geom:")==0) {
	  fscanf(config, "%s", input);
	  if (parse_geom(dxr2_info, input))
	    ;
	  else {
	    destroy_interface();
	    print_error("ERROR in config file %s, argument %s is invalid!", config_file, input);
	    exit(1);
	  }
	}
	else if(strcmp(input, "overlay_crop:")==0) {
	  fscanf(config, "%s", input);
	  if (parse_crop(dxr2_info, input))
	    ;
	  else {
	    destroy_interface();
	    print_error("ERROR in config file %s, argument %s is invalid!", config_file, input);
	    exit(1);
	  }
	}
      }
    }
  }
}

void open_files(player_info_t* player_info)
{
  open_new_vob(player_info, player_info->file_name);

  // open DVD device
  if ((player_info->dxr2FD = open(player_info->dxr2_device, O_WRONLY)) < 0) {
    
    print_error("ERROR: Cannot open DVD device (%s): %s\n", "/dev/dxr2", strerror(errno));
    wait_if_needed();
    cleanup_and_exit(player_info);
  }  

  // open drive
  if (player_info->encrypted) {

    if ((player_info->driveFD = open(player_info->dvd_device, O_RDONLY)) < 0) {
      
      print_error("ERROR: Cannot open DVD drive (%s): %s\n", player_info->dvd_device, strerror(errno));
      wait_if_needed();
      cleanup_and_exit(player_info);
    }  
  }
}

int open_new_vob(player_info_t* player_info, char* file_name)
{
  char** names;

  if(file_name != player_info->file_name) {
    print_info("setting file name to %s", file_name);
    strcpy(player_info->file_name, file_name);
  }

  names = generate_filenames(file_name);

  if (player_info->file_name[0] == '-') {
    print_error("using stdin for mpeg file\n");
    player_info->mpegFD= STDIN_FILENO;
  }
  else if(player_info->title && (player_info->mpegFD = open_vob(file_name)) < 0) {
    //    player_info->title = 0;
      print_error("ERROR: Could not open VOB title.  Try it without -T\n");
      wait_if_needed();
      cleanup_and_exit(player_info);    
  }
  else if ((player_info->mpegFD = multi_open(names)) < 0) {
      print_error("ERROR: Could not open mpeg (%s): %s\n", player_info->file_name, strerror(errno));
      wait_if_needed();
      cleanup_and_exit(player_info);
  }
}


void install_firmware(player_info_t* player_info)
{
  int uCodeFD;
  int uCodeSize;
  dxr2_uCode_t* uCode;

  if ((uCodeFD = open(player_info->uCode_file, O_RDONLY)) < 0) {
    
    print_error("ERROR: Could not open uCode (%s): %s\n", player_info->uCode_file, strerror(errno));
    wait_if_needed();
    cleanup_and_exit(player_info);
  }
  uCodeSize = lseek(uCodeFD, 0, SEEK_END);
  if ((uCode = malloc(uCodeSize + 4)) == NULL) {
    
    print_error("ERROR: Could not allocate memory for uCode: %s\n", strerror(errno));
    wait_if_needed();
    cleanup_and_exit(player_info);
  }
  lseek(uCodeFD, 0, SEEK_SET);
  if (read(uCodeFD, uCode+4, uCodeSize) != uCodeSize) {
    
    print_error("ERROR: Could not read uCode uCode: %s\n", strerror(errno));
    wait_if_needed();
    cleanup_and_exit(player_info);
  }
  close(uCodeFD);
  uCode->uCodeLength = uCodeSize;

  // upload ucode
  if (dxr2_install_firmware(player_info->dxr2FD, uCode)) {
    
    print_error("uCode upload failed!\n");
    wait_if_needed();
    cleanup_and_exit(player_info);
  }
}

void authenticate_disc(player_info_t* player_info)
{
    // OK, do the authentication
    if (css_do_disc_key(player_info->driveFD, player_info->dxr2FD, &player_info->authBuf)) {
      
      print_error("CSS authentication failed - cannot get disk key\n");
      wait_if_needed();
      cleanup_and_exit(player_info);
    }

    // invalidate AGID
    css_invalidate_AGID(player_info->driveFD, &player_info->authBuf);

    // get LBA
    if (css_get_lba(player_info->file_name, &player_info->lba)) {
      
      print_error("CSS authentication failed - cannot get LBA %s\n");
      wait_if_needed();
      cleanup_and_exit(player_info);
    }

    // do the title key!
    if (css_do_title_key(player_info->driveFD, player_info->dxr2FD, &player_info->authBuf, player_info->lba)) {
      
      print_error("CSS authentication failed - cannot get title key\n");
      wait_if_needed();
      cleanup_and_exit(player_info);
    }
}

int parse_geom(dxr2_status_info_t* dxr2_info, char* geom)
{
  int width,height,xoffset,yoffset;

  width=height=xoffset=yoffset=0;

  if(sscanf(geom, "%dx%d+%d+%d", &width, &height, &xoffset, &yoffset) != 3) {
    dxr2_info->overlay_pos.arg1=xoffset;
    dxr2_info->overlay_pos.arg2=yoffset;
    dxr2_info->overlay_dim.arg1=width;
    dxr2_info->overlay_dim.arg2=height;
    return 1;
  }
  return 0;
}

int parse_crop(dxr2_status_info_t* dxr2_info, char* crop)
{
  int left,right,top,bottom;

  if(sscanf(crop, "%dx%dx%dx%d", &left,&right,&top,&bottom) != 3) {
    print_info("CROP: %dx%dx%dx%d\n", left,right,top,bottom);
    dxr2_info->overlay_crop.arg1=left;
    dxr2_info->overlay_crop.arg2=right;
    dxr2_info->overlay_crop.arg3=top;
    dxr2_info->overlay_crop.arg4=bottom;
    return 1;
  }
  return 0;
}




