/*
  **********************************************************************
  *
  *     Copyright 1999, 2000 Creative Labs, Inc.
  *
  **********************************************************************
  *
  *     Date                 Author               Summary of changes
  *     ----                 ------               ------------------
  *     October 20, 1999     Andrew de Quincey    Rewrote and extended
  *                          Lucien Murray-Pitts  original incomplete 
  *                                               driver.
  *
  *     April 18, 1999       Andrew Veliath       Original Driver
  *                                               implementation
  *
  **********************************************************************
  *
  *     This program is free software; you can redistribute it and/or
  *     modify it under the terms of the GNU General Public License as
  *     published by the Free Software Foundation; either version 2 of
  *     the License, or (at your option) any later version.
  *
  *     This program is distributed in the hope that it will be useful,
  *     but WITHOUT ANY WARRANTY; without even the implied warranty of
  *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  *     GNU General Public License for more details.
  *
  *     You should have received a copy of the GNU General Public
  *     License along with this program; if not, write to the Free
  *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
  *     USA.
  *
  **********************************************************************
  */



/*
 **************************************************************************
 **************************************************************************
 * And now, a quick note about CSS and this software:                     *
 * 									  *
 **************************************************************************
 * Although this program permits playing of encrypted DVDs, 		  *
 * there are no "CSS secrets" contained in it, and none were used in 	  *
 * it's construction. No decrypted keys/data are handled by this program  *
 * AT ANY TIME. The DXR2 card does all the CSS decryption on-board, 	  *
 * in hardware. All this program does is permit the exchange of 	  *
 * *encrypted* keys and *encrypted* data between the drive and 		  *
 * DXR2 card's hardware.						  *
 **************************************************************************
 *									  *
 * The authors are NOT connected, in any way, with any of the 		  *
 * "software CSS decryption" programs out there, and *DO NOT* endorse 	  *
 * their use in any way. Piracy is a crime!				  *
 *									  *
 * Just wanted to make that ABSOLUTELY CLEAR. 				  *
 * 									  *
 **************************************************************************
 **************************************************************************
 */




/**
 *
 * test program for playing VCDs/DVDs etc
 *
 */


#include <stdlib.h>
#include <dxr2.h>
#include <fcntl.h>
#include <unistd.h>
#include <pthread.h> 
#include <signal.h>
#include "css.h"
#include "player.h"
#include "config.h"
#include "interface.h"
#include "bookmarks.h"
#include "ifo.h"
#include "vobtitle.h"
#include "multifile.h"

void invalidate_win();

static int quit;

void* read_data_proc(void*);
void* write_data_proc(void*);
void* command_proc(void*);
void* interface_proc(void*);

void clear_buffer(player_info_t*);

int bookmark_command(char, player_info_t*);

int main(int argc, char* argv[]) 
{
  player_info_t player_info;
  pthread_t read_thread, write_thread, command_thread, interface_thread;
  int interactive_mode=1;
  dxr2_status_info_t dxr2_info;
  pthread_attr_t thread_attrib;

  init_interface(&player_info, &dxr2_info);
  initialize(argc, argv, &player_info, &dxr2_info);

  // enter play mode!
  dxr2_set_playmode(DXR2_PLAYMODE_PLAY);

  if(player_info.file_name[0] == '-')
    interactive_mode = 0;

  // signal to threads that it's time to die if this is 1;
  quit = 0;

  // Initialize thread stuff.
  pthread_mutex_init(&(player_info.mutex), NULL);
  pthread_cond_init(&(player_info.playing), NULL);
  pthread_cond_init(&(player_info.slots_free), NULL);
  pthread_cond_init(&(player_info.slots_full), NULL);

  // set up some player status stuff.
  player_info.status = DXR2_PLAYMODE_PLAY;
  player_info.frames_in_buffer = 0;
 
  redraw_window();

  pthread_attr_init(&thread_attrib);
  pthread_attr_setschedpolicy(&thread_attrib, SCHED_RR);
 
  pthread_create( &read_thread, &thread_attrib, read_data_proc, (void*) &player_info);
  pthread_create( &write_thread, &thread_attrib, write_data_proc, (void*) &player_info);

  if(interactive_mode)
    pthread_create( &command_thread, &thread_attrib, command_proc, (void*) &player_info);

  pthread_join(read_thread, NULL);
  pthread_join(write_thread, NULL);

  if(interactive_mode)
    pthread_kill(command_thread, SIGINT);

  cleanup_and_exit(&player_info);
}

void* read_data_proc(void* arg)
{
  player_info_t* player_info = (player_info_t*) arg;
  int size=1;
  int ourLoc = 0;
  frame buffer;
  int scan_counter=0;

  while(size != 0 && size != -1 && !quit) {
    pthread_mutex_lock(&(player_info->mutex));
    
    // If we're not playing, wait until we are.
    // if we're single stepping, just read one frame.
    if(player_info->status == DXR2_PLAYMODE_PAUSED || player_info->status == DXR2_PLAYMODE_SINGLESTEP) {
      pthread_cond_wait(&(player_info->playing), &(player_info->mutex));
    }

    // If the array of buffered frames is full, wait till one is empty.
    if(player_info->frames_in_buffer == NUM_BUFFERED_FRAMES) {
      pthread_cond_wait(&(player_info->slots_free), &(player_info->mutex));
    }
    pthread_mutex_unlock(&(player_info->mutex));
    
    if(player_info->status == DXR2_PLAYMODE_STOPPED) {
      print_info("quitting read proc...\n");
      pthread_exit(NULL);
    }

    if(player_info->clear_read_buffer) {
      ourLoc = 0;
      player_info->clear_read_buffer = 0;
    }

    size=-1;

    if(player_info->title)
      size = vob_read(buffer, FRAME_SIZE);
    else
      size = multi_read(buffer, FRAME_SIZE);

    if(size != 0) {
      pthread_mutex_lock(&(player_info->mutex));
      memcpy(player_info->frames[ourLoc], buffer, size);
      player_info->sizes[ourLoc] = size;
    }

    if(size != 0 && size != -1) {
      ++(player_info->frames_in_buffer);
      ourLoc = (ourLoc + 1) % NUM_BUFFERED_FRAMES;
      pthread_cond_signal(&(player_info->slots_full));
    }

    pthread_mutex_unlock(&(player_info->mutex));    
  }

  quit = 1;

  print_info("quitting read proc\n");
}

void* write_data_proc(void* arg)
{
  player_info_t* player_info = (player_info_t*) arg;
  int size=1;
  int ourLoc = 0;
  int tics,secs,mins,hrs;
  frame buffer;

  tics=secs=mins=hrs=0;

  while(!quit) {
    pthread_mutex_lock(&(player_info->mutex));

    // If we're not playing, wait until we are.
    if(player_info->status == DXR2_PLAYMODE_PAUSED) 
      pthread_cond_wait(&(player_info->playing), &(player_info->mutex));

    if(player_info->status == DXR2_PLAYMODE_SINGLESTEP) {
      player_info->status = DXR2_PLAYMODE_PAUSED;
      dxr2_set_playmode(player_info->status);
    }

    //    if(player_info->status == DXR2_PLAYMODE_SCAN && skip==0)
    //      skip = 35;
  
    // If there are no frames in the buffer, wait until there are.
    if(player_info->frames_in_buffer == 0)
      pthread_cond_wait(&(player_info->slots_full), &(player_info->mutex));

    // If we've quit while waiting, stop.
    if(player_info->status == DXR2_PLAYMODE_STOPPED) {
      pthread_mutex_unlock(&(player_info->mutex));
      pthread_cond_signal(&(player_info->slots_free));
      pthread_exit(NULL);
    }

    if(player_info->clear_write_buffer) {
      ourLoc = 0;
      player_info->clear_write_buffer = 0;
    }

    size = player_info->sizes[ourLoc];

    // don't write a null frame when switching files.
    if(size != 0 && size != -1) {
      memcpy(buffer, player_info->frames[ourLoc], size);
      ourLoc = (ourLoc+1) % NUM_BUFFERED_FRAMES;
      --(player_info->frames_in_buffer);
      pthread_cond_signal(&(player_info->slots_free));
    }

    pthread_mutex_unlock(&(player_info->mutex));

    if(size != 0 && size != -1)
      write(player_info->dxr2FD, buffer, size);

    tics = (tics+2)%30;
    if(tics==0) {
      secs = (secs+1) % 60;
      if(secs==0) {
	mins = (mins+1) % 60;
	if(mins==0)
	  ++hrs;
      }
    }
  }
  print_info("quitting write proc\n");
}

void* command_proc(void* arg)
{
  player_info_t* player_info = (player_info_t*) arg;
  char c;
  int done=0;
  int newmode;
  int bookmark_commands = 0;
  int chapter;
  char input[80];

  while(!done && !quit) {
    c=get_a_char();

    if(quit)
      c='q';

    if(bookmark_commands)
      bookmark_commands = bookmark_command(c, player_info);
    else
      switch (c) {
      case 'q':
      case 'Q':
      case 's':
	//      case 'S':
	player_info->status = DXR2_PLAYMODE_STOPPED;
	quit = done = 1;
	// unfreeze processes if we're paused.
	pthread_cond_broadcast(&(player_info->playing));
	// free up read if it's waiting on empty slots.
	pthread_cond_broadcast(&(player_info->slots_free));
	// free up write if it's waiting on full slots.
	pthread_cond_broadcast(&(player_info->slots_full));
	invalidate_win();
	break;
      case 'p':
      case 'P':
	// play/pause
	pthread_mutex_lock(&(player_info->mutex));
	if(player_info->status != DXR2_PLAYMODE_PAUSED) {
	  player_info->old_status = player_info->status;
	  player_info->status = DXR2_PLAYMODE_PAUSED;
	} else {
	  player_info->status = player_info->old_status;
	  pthread_cond_broadcast(&(player_info->playing));
	}
	dxr2_set_playmode(player_info->status);
	pthread_mutex_unlock(&(player_info->mutex));
	invalidate_win();
	break;
      case 'h':
      case 'H':
	print_info("Commands:\n");
	print_info("   's' == Stop\n");
	print_info("   'q' == Quit\n");
	print_info("   'p' == Pause/Resume\n");
	print_info("   'f' == Single Step forwards(when paused)\n");
	print_info("   'n' == Next chapter\n");
	print_info("   'v' == preVious chapter\n");
	print_info("   'c' == Choose chapter\n");
	print_info("   'b' == Bookmark mode\n");
	//	print_info("   'o' == toggle VGA Overlay\n");
	print_info("   'o' == DISABLED ATM: toggle VGA Overlay\n");
	print_info("   'h' == Help ( this listing )\n");
	break;
      case 'b':
      case 'B':
	print_info("Bookmark commands.\n");
	print_info("'h' for help.\n");
	bookmark_commands = 1;
	break;
      case 'n':
      case 'N':
	vob_next_chapter();
	clear_buffer(player_info);
	break;
      case 'v':
      case 'V':
	vob_prev_chapter();
	clear_buffer(player_info);
	break;
      case 'c':
      case 'C':
	print_info("Which chapter would you like?  [1-%d]", vob_num_chapters());
	get_a_str(input, 80);
	input[strlen(input)] = '\0';
	sscanf(input, "%d", &chapter);
	print_info("Seeking chapter: %d\n", chapter);
	--chapter; // convert from 1-based indexing to 0-based indexing
	vob_chapter_seek(chapter);
	clear_buffer(player_info);
	break;
      case 'o':
      case 'O':
	//        toggle_overlay(player_info);
        break;	
      case 'f':
	// go forward
	pthread_mutex_lock(&(player_info->mutex));
	if(player_info->status == DXR2_PLAYMODE_PAUSED) {
	  player_info->status = DXR2_PLAYMODE_SINGLESTEP;
	  dxr2_set_playmode(player_info->status);
	  pthread_cond_broadcast(&(player_info->playing));
	}
	pthread_mutex_unlock(&(player_info->mutex));
	invalidate_win();
	break;
	/*
	  case 'r':
	  // reverse play
	  pthread_mutex_lock(&(player_info->mutex));
	  player_info->status = DXR2_PLAYMODE_REVERSE_PLAY;
	  dxr2_set_playmode(player_info->status);
	  pthread_mutex_unlock(&(player_info->mutex));
	  break;
      case 'S':
	// Slow-Mo
	pthread_mutex_lock(&(player_info->mutex));
	player_info->status = DXR2_PLAYMODE_PAUSED;
	dxr2_set_playmode(player_info->status);
	player_info->status = DXR2_PLAYMODE_SLOWFORWARDS;
	dxr2_set_playmode(player_info->status);
	pthread_mutex_unlock(&(player_info->mutex));
	break;
	*/
	
      case 'a':
      case 'A': // fAst forward
	pthread_mutex_lock(&(player_info->mutex));
	if(player_info->status == DXR2_PLAYMODE_SCAN) {
	  player_info->status = DXR2_PLAYMODE_PLAY;
	  dxr2_mute(DXR2_AUDIO_MUTE_OFF);
	} else {
	  player_info->status = DXR2_PLAYMODE_SCAN;
	  dxr2_mute(DXR2_AUDIO_MUTE_ON);
	}
	dxr2_set_playmode(player_info->status);
	pthread_mutex_unlock(&(player_info->mutex));
	invalidate_win();
	break;
      }
  }
  print_info("quitting command proc\n");
  pthread_exit(NULL);
}

int bookmark_command(char c, player_info_t* player_info)
{
  int idx;
  offset_t offset;

  switch(c) {
  case 'q':
  case 'Q':
    print_info("Leaving Bookmark Mode.\n");
    return 0;
    break;
  case 'p':
  case 'P':
    print_bookmarks();
    break;
  case 'c':
  case 'C':
    offset = 0;
    print_info("getting offset...");
    pthread_mutex_lock(&(player_info->mutex));
    if(player_info->title)
      offset = vob_lseek(0, SEEK_CUR);
    else
      offset = multi_lseek(0, SEEK_CUR);
    pthread_mutex_unlock(&(player_info->mutex));
    print_info("done.\n");
    create_bookmark_at(offset);
    break;
  case 'g':
  case 'G':
    idx = choose_bookmark();
    if(idx >= 0) {
      print_info("Bookmark %d chosen.\n", idx);
      offset = bookmark_offset(idx);
      pthread_mutex_lock(&(player_info->mutex));
      print_info("Seeking to ofset %d\n", offset);
      if(player_info->title)
	vob_lseek(offset, SEEK_SET);
      else
	multi_lseek(offset, SEEK_SET);
      pthread_mutex_unlock(&(player_info->mutex));
    }
    clear_buffer(player_info);
    break;
  case 'l':
  case 'L':
    print_info("What file do you want to load?\n\t");
    get_a_str(player_info->bookmark_file, 80);
    load_bookmarks(player_info->bookmark_file);
    break;
  case 's':
  case 'S':
    if(player_info->bookmark_file[0] == '\0') {
      print_info("What should I save the file as?\n\t");
      get_a_str(player_info->bookmark_file, 80);
    }
    write_bookmarks(player_info->bookmark_file);
    break;
  case 'h':
  case 'H':
    print_info("Bookmark Help:\n");
    print_info("\n");
    print_info("  'p' == Print Bookmarks\n");
    print_info("  'c' == Create a new bookmark at the present location\n");
    print_info("  'g' == Goto a bookmark\n");
    print_info("  'l' == Load Bookmarks\n");
    print_info("  's' == Save Bookmarks\n");
    break;
  }
  return 1;
}
  
void cleanup_and_exit(player_info_t* player_info)
{
  destroy_interface();
  // since we've finished, invalidate the AGID!
  if (player_info->driveFD != -1) {
    // invalidate AGID
    css_invalidate_AGID(player_info->driveFD, &player_info->authBuf);
  }
 
  // close it again
  if(player_info->dxr2FD != -1) {
    dxr2_set_playmode(DXR2_PLAYMODE_STOPPED);
    close(player_info->dxr2FD);
  }

  if(player_info->mpegFD != STDIN_FILENO)
    close(player_info->mpegFD);

  if (player_info->driveFD != -1)
    close(player_info->driveFD);

  exit(0);
}

void clear_buffer(player_info_t* player_info)
{
  pthread_mutex_lock(&(player_info->mutex));
  player_info->frames_in_buffer = 0;
  player_info->clear_read_buffer = 1;
  player_info->clear_write_buffer = 1;
  pthread_mutex_unlock(&(player_info->mutex));
  pthread_cond_signal(&(player_info->slots_free));
}

void invalidate_win()
{
  
  redraw_window();
}


int toggle_overlay(player_info_t* player_info)
{
  dxr2_threeArg_t buf3;

  player_info->do_overlay = !player_info->do_overlay;
  if (player_info->do_overlay == 1)
    dxr2_set_overlay_mode(DXR2_OVERLAY_WINDOW_KEY);
  else
    dxr2_set_overlay_mode(DXR2_OVERLAY_DISABLED);

}
