/*
 **********************************************************************
 *     audio.c -- /dev/dsp interface for emu10k1 driver
 *     Copyright 1999, 2000 Creative Labs, Inc.
 *
 **********************************************************************
 *
 *     Date                 Author          Summary of changes
 *     ----                 ------          ------------------
 *     October 20, 1999     Bertrand Lee    base code release
 *     November 2, 1999	    Alan Cox        cleaned up types/leaks
 *
 **********************************************************************
 *
 *     This program is free software; you can redistribute it and/or
 *     modify it under the terms of the GNU General Public License as
 *     published by the Free Software Foundation; either version 2 of
 *     the License, or (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public
 *     License along with this program; if not, write to the Free
 *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 *     USA.
 *
 **********************************************************************
 */

/*
 *	FIXME:
 *		There are some 32bit assumptions here
 *		Cast of pointers to 32bit values is unsafe
 *			- Wave callbacks and elsewhere
 */

#define __NO_VERSION__
#include <linux/module.h>

#include "hwaccess.h"
#include "mycommon.h"
#include "mmwave.h"
#include "cardwo.h"
#include "cardwi.h"

#define MINFRAGS	2
#define MINFRAGSHIFT	4

struct sblive_wavedevice 
{
	struct sblive_hw *sb_hw;
	struct wiinst *wiinst;
	struct woinst *woinst;
	u16 enablebits;
};

static int waveOutCallbackFn(unsigned long, unsigned long, unsigned long);
static int waveInCallbackFn(unsigned long, unsigned long, unsigned long);
static void calculate_ofrag(struct woinst *woinst);
static void calculate_ifrag(struct wiinst *wiinst);

// audio file operations
#if LINUX_VERSION_CODE < 0x020100
static int emu10k1_audio_llseek(struct inode *inode, struct file *file, off_t offset, int nOrigin)
#else
static loff_t emu10k1_audio_llseek(struct file *file, loff_t offset, int nOrigin)
#endif
{
	PDEBUG("emu10k1_audio_lseek() called\n");
	return -ESPIPE;
}

#if LINUX_VERSION_CODE < 0x020100
static int emu10k1_audio_read(struct inode *inode, struct file *file, char *buffer, int count)
#else
static ssize_t emu10k1_audio_read(struct file *file, char *buffer, size_t count, loff_t * ppos)
#endif
{
	struct sblive_wavedevice * wave_dev = (struct sblive_wavedevice *) file->private_data;
	struct wiinst *wiinst = wave_dev->wiinst;
	struct wave_in *wave_in = wiinst->wave_in;
	ssize_t ret = 0;
	u32 flags;

	spin_lock_irqsave(&wiinst->lock, flags);

	PDEBUG("emu10k1_audio_read() called, buffer=%#x, count=%d\n", (unsigned int) buffer, count);

	if (file->f_mode & FMODE_READ) {
		if (!wave_in) {
			calculate_ifrag(wiinst);
			if (sblive_waveinOpen(wave_dev->sb_hw,
					   &wiinst->wave_fmt,
					   waveInCallbackFn,
					   (u32) wave_dev,
					   &wiinst->fragment_size,
					   wiinst->numfrags,
					   & wiinst->wave_in) != CTSTATUS_SUCCESS) {
				DPF("   sblive_waveinOpen failed!");
				spin_unlock_irqrestore(&wiinst->lock, flags);
				return -EINVAL;
			}
			DPD("   fragment_size size is -> ", wiinst->fragment_size);
		}

		wave_in = wiinst->wave_in;
		if (!access_ok(VERIFY_WRITE, buffer, count)){
			spin_unlock_irqrestore(&wiinst->lock, flags);
			return -EFAULT;
		}

		while (count > 0) {
			u32 bytestocopy, pending;
			int status;

			if (wave_dev->enablebits & PCM_ENABLE_INPUT
			    && wave_in->state != CARDWAVE_STATE_STARTED) {
				if (sblive_waveinStart(wave_dev->sb_hw, wave_in) != CTSTATUS_SUCCESS) {
					PDEBUG("sblive_waveinStart failed.\n");
					spin_unlock_irqrestore(&wiinst->lock, flags);
					return -EFAULT;
				}
				DPF("    recording started.");
			}
                        status = sblive_waveinGetXferSize(wave_dev->sb_hw->card_wavein,
						       wave_in,
						       &bytestocopy, &pending);
			if (status != CTSTATUS_SUCCESS) {
				PDEBUG("sblive_waveinGetXferSize failed\n");
				spin_unlock_irqrestore(&wiinst->lock, flags);
				return -EFAULT;
			}

			DPD("    bytestocopy --> ", bytestocopy);

			if (bytestocopy < count && bytestocopy < wiinst->fragment_size) {
				if ((file->f_flags & O_NONBLOCK)
				    | (!(wave_dev->enablebits & PCM_ENABLE_INPUT))) {
					//          DPD("    return -> ", ret);
					spin_unlock_irqrestore(&wiinst->lock, flags);
					return (ret ? ret : -EAGAIN);
				}
				DPF("    Go to Sleep...");
				spin_unlock_irqrestore(&wiinst->lock, flags);
				interruptible_sleep_on(&wave_in->wait_queue);
				if (signal_pending(current))
					return (ret ? ret : -ERESTARTSYS);

				spin_lock_irqsave(&wiinst->lock, flags);
				continue;
			}
			bytestocopy = min(bytestocopy, count);
			status = sblive_waveinXferData(wave_dev->sb_hw->card_wavein,
						    wave_in,
						    (u8 *) buffer,
						    &bytestocopy);
			if (status != CTSTATUS_SUCCESS) {
				PDEBUG("sblive_waveinXferData failed.\n");
				spin_unlock_irqrestore(&wiinst->lock, flags);
				return ret ? ret : -EFAULT;
			}
			count -= bytestocopy;
			buffer += bytestocopy;
			ret += bytestocopy;
		}
		DPD("    bytes copied -> ", ret);
	}
	spin_unlock_irqrestore(&wiinst->lock, flags);
	return ret;
}

#if LINUX_VERSION_CODE < 0x020100
static int emu10k1_audio_write(struct inode *inode, struct file *file, const char *buffer, int count)
#else
static ssize_t emu10k1_audio_write(struct file *file, const char *buffer, size_t count, loff_t * ppos)
#endif
{
	struct sblive_wavedevice *wave_dev = (struct sblive_wavedevice *) file->private_data;
	struct woinst *woinst = wave_dev->woinst;
	struct wave_out *wave_out = woinst->wave_out;
	ssize_t ret;
	u32 flags;
#if LINUX_VERSION_CODE >= 0x020100	
	struct inode *inode =  file->f_dentry->d_inode; 
#endif	

	spin_lock_irqsave(&woinst->lock, flags);

	PDEBUG("emu10k1_audio_write() called, buffer=%#x, count=%d\n", (unsigned int) buffer, count);

	if (woinst->mapped){
		spin_unlock_irqrestore(&woinst->lock, flags);
		return -ENXIO;
	}

	if (!wave_out) {
		calculate_ofrag(woinst);
		while (sblive_waveoutOpen(wave_dev->sb_hw,
				   &woinst->wave_fmt,
				   waveOutCallbackFn,
				   (u32) wave_dev,
				   &woinst->fragment_size,
				   woinst->numfrags,
				   &woinst->wave_out) != CTSTATUS_SUCCESS) {
			if (file->f_flags & O_NONBLOCK){
				spin_unlock_irqrestore(&woinst->lock, flags);
				return -EAGAIN;
			}
#if LINUX_VERSION_CODE < 0x020307
			up(&inode->i_sem);
#endif
			spin_unlock_irqrestore(&woinst->lock, flags);
			interruptible_sleep_on(&wave_dev->sb_hw->open_wait);
#if LINUX_VERSION_CODE < 0x020307
			down(&inode->i_sem);
#endif
			if (signal_pending(current))
				return -ERESTARTSYS;
			spin_lock_irqsave(&woinst->lock, flags);
		}
		wave_out = woinst->wave_out;
	}

	if (!access_ok(VERIFY_READ, buffer, count)){
		spin_unlock_irqrestore(&woinst->lock, flags);
		return -EFAULT;
	}

	ret = 0;
	while (count > 0) {
		u32 bytestocopy, pending;
		int status;

		status = sblive_waveoutGetXferSize(wave_dev->sb_hw->card_waveout,
						   wave_out,
						   &bytestocopy,
						   &pending);
               if (status != CTSTATUS_SUCCESS) {
                        PDEBUG("sblive_waveoutGetXferSize failed\n");
                        spin_unlock_irqrestore(&woinst->lock, flags);
                        return -EFAULT;
                }

		if(woinst->silence_filled > 0){
			if(woinst->silence_filled == 1){
				if(pending <= woinst->fragment_size)
                        		woinst->silence_filled = 2;
				else
					bytestocopy += woinst->fragment_size;
			}
                	if(woinst->silence_filled == 2)	
                        	bytestocopy = woinst->fragment_size * woinst->numfrags;
		}

		DPD("    bytestocopy --> ", bytestocopy);
		if (bytestocopy < woinst->fragment_size && bytestocopy < count) {
			if ((file->f_flags & O_NONBLOCK)
			    | (!(wave_dev->enablebits & PCM_ENABLE_OUTPUT))) {
				//      DPD("    return -> ", ret);
				spin_unlock_irqrestore(&woinst->lock, flags);
				return (ret ? ret : -EAGAIN);
			}
			DPF("    Go to Sleep...");
#if LINUX_VERSION_CODE < 0x020307
			up(&inode->i_sem);
#endif
			spin_unlock_irqrestore(&woinst->lock, flags);
			interruptible_sleep_on(&woinst->wait_queue);
#if LINUX_VERSION_CODE < 0x020307
			down(&inode->i_sem);
#endif
			if (signal_pending(current))
				return (ret ? ret : -ERESTARTSYS);
			spin_lock_irqsave(&woinst->lock, flags);	
			continue;
		}
		bytestocopy = min(bytestocopy, count);
		if(woinst->silence_filled > 0){
			u32 tmp;
			if(woinst->silence_filled == 1)
				tmp = woinst->silence_start;
			else {
                        	sblive_waveoutGetControl(wave_dev->sb_hw->card_waveout,
                                	                 wave_out,
                                        	         WAVECURPOS,
                                                	 &tmp);
				pending = 0;
			}

			if(pending + bytestocopy < woinst->fragment_size){
				woinst->silence_filled = 1;
				woinst->silence_start = tmp + bytestocopy;
				if(woinst->silence_start > woinst->fragment_size * woinst->numfrags)
					woinst->silence_start -= woinst->fragment_size * woinst->numfrags;
			}
			else
				woinst->silence_filled = 0;
                        sblive_waveoutSetControl(wave_dev->sb_hw,
                                                 wave_out,
                                                 WAVEWRITEPOINTER,
                                                 &tmp);
                }
		status = sblive_waveoutXferData(wave_dev->sb_hw->card_waveout,
					    wave_out,
					    (u8 *) buffer,
					    &bytestocopy);

		if (woinst->silence_filled) {
			u32 tmp = woinst->fragment_size;
			sblive_waveoutFillSilence(wave_dev->sb_hw->card_waveout,
					       wave_out,
					       &tmp);
		}

		if (status != CTSTATUS_SUCCESS) {
			PDEBUG("sblive_waveoutXferData failed.\n");
			spin_unlock_irqrestore(&woinst->lock, flags);
			return ret ? ret : -EFAULT;
		}
		count -= bytestocopy;
		buffer += bytestocopy;
		ret += bytestocopy;
		woinst->total_copied += bytestocopy;
		if (wave_dev->enablebits & PCM_ENABLE_OUTPUT
		    && wave_out->state != CARDWAVE_STATE_STARTED
		    && woinst->total_copied >= woinst->fragment_size) {
			if (sblive_waveoutStart(wave_dev->sb_hw, wave_out) != CTSTATUS_SUCCESS) {
				PDEBUG("sblive_waveoutStart failed.\n");
				spin_unlock_irqrestore(&woinst->lock, flags);
				return -EFAULT;
			}
		}
	}
	spin_unlock_irqrestore(&woinst->lock, flags);
	DPD("    bytes copied -> ", ret);
	return ret;
}

static int emu10k1_audio_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	struct sblive_wavedevice *wave_dev = (struct sblive_wavedevice *) file->private_data;
	int val = 0;

	PDEBUG("emu10k1_audio_ioctl() called:\n");
	switch (cmd) {
	case OSS_GETVERSION:
		PDEBUG("OSS_GETVERSION:\n");
		return put_user(SOUND_VERSION, (int *) arg);

	case SNDCTL_DSP_RESET:
		PDEBUG("SNDCTL_DSP_RESET:\n");
		wave_dev->enablebits = PCM_ENABLE_OUTPUT | PCM_ENABLE_INPUT;
		if (file->f_mode & FMODE_WRITE) {
			struct woinst *woinst = wave_dev->woinst;
			struct wave_out * wave_out = woinst->wave_out;
			u32 flags, dummy;
			spin_lock_irqsave(&woinst->lock, flags);
			if (wave_out) {
				sblive_waveoutStop(wave_dev->sb_hw,
					       wave_out,
					       &dummy);
				wave_out->wavexferbuf->xferpos = 0;
				wave_out->wavexferbuf->stopposition = 0;
			}
			woinst->total_copied = 0;
			woinst->total_played = 0;
			woinst->silence_filled = 0;
			woinst->silence_start = 0;
			woinst->getoptr_blocks = 0;
			woinst->wave_ptr = 0;
			spin_unlock_irqrestore(&woinst->lock, flags);
		}
		if (file->f_mode & FMODE_READ) {
			struct wiinst *wiinst = wave_dev->wiinst;
			struct wave_in *wave_in = wiinst->wave_in;
			u32 flags, dummy;
			spin_lock_irqsave(&wiinst->lock, flags);
			if (wave_in)
				sblive_waveinStop(wave_dev->sb_hw,
					       wave_in,
					       &dummy);
			spin_unlock_irqrestore(&wiinst->lock, flags);
		}
		break;

	case SNDCTL_DSP_SYNC:
		PDEBUG("SNDCTL_DSP_SYNC:\n");
		if (file->f_mode & FMODE_WRITE) {
			struct woinst *woinst = wave_dev->woinst;
			struct wave_out * wave_out = woinst->wave_out;
			u32 flags;

			spin_lock_irqsave(&woinst->lock, flags);
			if (wave_out && wave_out->state == CARDWAVE_STATE_STARTED)
				while ((woinst->total_played < woinst->total_copied)
				       & !signal_pending(current)){
					spin_unlock_irqrestore(&woinst->lock, flags);
					interruptible_sleep_on(&woinst->wait_queue);
					spin_lock_irqsave(&woinst->lock, flags);
				}
			spin_unlock_irqrestore(&woinst->lock, flags);
		}
		break;

	case SNDCTL_DSP_SETDUPLEX:
		PDEBUG("SNDCTL_DSP_SETDUPLEX:\n");
		break;

	case SNDCTL_DSP_GETCAPS:
		PDEBUG("SNDCTL_DSP_GETCAPS:\n");
		return put_user(DSP_CAP_DUPLEX | DSP_CAP_REALTIME | DSP_CAP_TRIGGER | DSP_CAP_MMAP, (int *) arg);

	case SNDCTL_DSP_SPEED:
		PDEBUG("SNDCTL_DSP_SPEED:\n");

		get_user_ret(val, (int *) arg, -EFAULT);
		DPD(" val is ", val);
		if (val >= 0) {
			if (file->f_mode & FMODE_WRITE) {
				struct woinst *woinst = wave_dev->woinst;
				struct wave_out *wave_out = woinst->wave_out;
				u32 flags, dummy;

				spin_lock_irqsave(&woinst->lock, flags);
				if (wave_out)
					sblive_waveoutStop(wave_dev->sb_hw,
						       wave_out,
						       &dummy);

				woinst->wave_fmt.samplingrate = val;

				if (woinst->mapped) {
					struct wave_format wave_fmt = woinst->wave_fmt;
					if (sblive_waveoutSetControl(wave_dev->sb_hw,
								     wave_out,
								     WAVEINSTANCEFORMAT,
								     (u32 *) &wave_fmt)
					    != CTSTATUS_SUCCESS){
						spin_unlock_irqrestore(&woinst->lock, flags);
						return -EINVAL;
					}
				}
				DPD("    set playback sampling rate -> ", woinst->wave_fmt.samplingrate);
				spin_unlock_irqrestore(&woinst->lock, flags);
			}
			if (file->f_mode & FMODE_READ) {
				struct wiinst *wiinst = wave_dev->wiinst;
				struct wave_in *wave_in = wiinst->wave_in;
				u32 flags, dummy;
				struct wave_format *wave_fmt;

				spin_lock_irqsave(&wiinst->lock, flags);
				if (wave_in)
					sblive_waveinStop(wave_dev->sb_hw,
							wave_in,
						       &dummy);
				wave_fmt = &wave_dev->wiinst->wave_fmt;
				wave_fmt->samplingrate = val;
				if (sblive_waveinQueryFormat(wave_dev->sb_hw->card_wavein,
							  wave_fmt,
							  CARDWAVE_QF_ALL)
				    != CTSTATUS_SUCCESS)
					val = wave_fmt->samplingrate = 8000;	// default
				DPD("    set recording sampling rate -> ", wave_fmt->samplingrate);
				spin_unlock_irqrestore(&wiinst->lock, flags);
			}
			return put_user(val, (int *) arg);
		} else {
			struct wave_format *wave_fmt = NULL;

			if (file->f_mode & FMODE_READ)
				wave_fmt = &wave_dev->wiinst->wave_fmt;
			else if (file->f_mode & FMODE_WRITE)
				wave_fmt = &wave_dev->woinst->wave_fmt;
			return put_user(wave_fmt->samplingrate, (int *) arg);
		}
		break;

	case SNDCTL_DSP_STEREO:
		PDEBUG("SNDCTL_DSP_STEREO:\n");

		get_user_ret(val, (int *) arg, -EFAULT);
		DPD(" val is ", val);
		if (file->f_mode & FMODE_WRITE) {
			struct woinst *woinst = wave_dev->woinst;
			struct wave_out *wave_out = woinst->wave_out;
			u32 flags, dummy;
			
			spin_lock_irqsave(&woinst->lock, flags);
			if (wave_out)
				sblive_waveoutStop(wave_dev->sb_hw,
					       wave_out,
					       &dummy);

			woinst->wave_fmt.channels = val ? 2 : 1;

			if (woinst->mapped) {
				struct wave_format wave_fmt = woinst->wave_fmt;
				if (sblive_waveoutSetControl(wave_dev->sb_hw,
							 wave_out,
							 WAVEINSTANCEFORMAT,
							 (u32 *) &wave_fmt) != CTSTATUS_SUCCESS){
					spin_unlock_irqrestore(&woinst->lock, flags);
					return -EINVAL;
				}
			}
			DPD("    set playback stereo -> ", woinst->wave_fmt.channels);
			spin_unlock_irqrestore(&woinst->lock, flags);
		}
		if (file->f_mode & FMODE_READ) {
			struct wiinst *wiinst = wave_dev->wiinst;
			struct wave_in *wave_in = wiinst->wave_in;
			u32 flags, dummy;
			struct wave_format *wave_fmt;

			spin_lock_irqsave(&wiinst->lock, flags);
			if (wave_in)
				sblive_waveinStop(wave_dev->sb_hw,
					       wave_in,
					       &dummy);
			wave_fmt = &wave_dev->wiinst->wave_fmt;
			wave_fmt->channels = val ? 2 : 1;
			if (sblive_waveinQueryFormat(wave_dev->sb_hw->card_wavein,
						  wave_fmt,
						  CARDWAVE_QF_ALL) != CTSTATUS_SUCCESS){
				spin_unlock_irqrestore(&wiinst->lock, flags);
				return -EINVAL;
			}
			DPD("    set recording stereo -> ", wave_fmt->channels);
			spin_unlock_irqrestore(&wiinst->lock, flags);
		}
		break;

	case SNDCTL_DSP_CHANNELS:
		PDEBUG("SNDCTL_DSP_CHANNELS:\n");

		get_user_ret(val, (int *) arg, -EFAULT);
		DPD(" val is ", val);
		if (val != 0) {
			if (file->f_mode & FMODE_WRITE) {
				struct woinst *woinst = wave_dev->woinst;
				struct wave_out *wave_out = woinst->wave_out;
				u32 flags, dummy;

				spin_lock_irqsave(&woinst->lock, flags);
				if (wave_out)
					sblive_waveoutStop(wave_dev->sb_hw,
						       wave_out, &dummy);

				woinst->wave_fmt.channels = val;

				if (woinst->mapped) {
					struct wave_format wave_fmt = woinst->wave_fmt;
					if (sblive_waveoutSetControl(wave_dev->sb_hw,
								 wave_out,
								 WAVEINSTANCEFORMAT,
								 (u32 *) &wave_fmt) != CTSTATUS_SUCCESS){
						spin_unlock_irqrestore(&woinst->lock, flags);
						return -EINVAL;
					}
					DPD("    set playback number of channels -> ", woinst->wave_fmt.channels);
				}
				spin_unlock_irqrestore(&woinst->lock, flags);
			}
			if (file->f_mode & FMODE_READ) {
				struct wiinst *wiinst = wave_dev->wiinst;
				struct wave_in *wave_in = wiinst->wave_in;
				u32 flags, dummy;
				struct wave_format *wave_fmt;

				spin_lock_irqsave(&wiinst->lock, flags);
				if (wave_in)
					sblive_waveinStop(wave_dev->sb_hw,
						       wave_in,
						       &dummy);
				wave_fmt = &wave_dev->wiinst->wave_fmt;
				wave_fmt->channels = val;
				if (sblive_waveinQueryFormat(wave_dev->sb_hw->card_wavein,
							  wave_fmt,
							  CARDWAVE_QF_ALL) != CTSTATUS_SUCCESS){
					spin_unlock_irqrestore(&wiinst->lock, flags);
					return -EINVAL;
				}
				DPD("    set recording number of channels -> ", wave_fmt->channels);
				spin_unlock_irqrestore(&wiinst->lock, flags);
			}
			return put_user(val, (int *) arg);
		} else {
			struct wave_format *wave_fmt = NULL;

			if (file->f_mode & FMODE_READ)
				wave_fmt = &wave_dev->wiinst->wave_fmt;
			else if (file->f_mode & FMODE_WRITE)
				wave_fmt = &wave_dev->woinst->wave_fmt;
			return put_user(wave_fmt->channels, (int *) arg);
		}
		break;

	case SNDCTL_DSP_GETFMTS:
		PDEBUG("SNDCTL_DSP_GETFMTS:\n");
		return put_user(AFMT_S16_LE | AFMT_U8, (int *) arg);

	case SNDCTL_DSP_SETFMT:	// Same as SNDCTL_DSP_SAMPLESIZE
		{
			PDEBUG("SNDCTL_DSP_SETFMT:\n");

			get_user_ret(val, (int *) arg, -EFAULT);
			DPD(" val is ", val);
			if (val != AFMT_QUERY) {
				if (file->f_mode & FMODE_WRITE) {
					struct woinst *woinst = wave_dev->woinst;
					struct wave_out *wave_out = woinst->wave_out;
					u32 flags, dummy;

					spin_lock_irqsave(&woinst->lock, flags);
					if (wave_out)
						sblive_waveoutStop(wave_dev->sb_hw,
								   wave_out,
								   &dummy);

					woinst->wave_fmt.bitspersample = val;

					if (woinst->mapped) {
						struct wave_format wave_fmt = woinst->wave_fmt;
						if (sblive_waveoutSetControl(wave_dev->sb_hw,
									     wave_out,
									     WAVEINSTANCEFORMAT,
									     (u32 *) &wave_fmt) != CTSTATUS_SUCCESS){
							spin_unlock_irqrestore(&woinst->lock, flags);
							return -EINVAL;
					}
					}
					DPD("    set playback sample size -> ", woinst->wave_fmt.bitspersample);
					spin_unlock_irqrestore(&woinst->lock, flags);
				}
				if (file->f_mode & FMODE_READ) {
					struct wiinst *wiinst = wave_dev->wiinst;
					struct wave_in *wave_in = wiinst->wave_in;
					u32 flags, dummy;
					struct wave_format *wave_fmt;
					
					spin_lock_irqsave(&wiinst->lock, flags);
					if (wave_in)
						sblive_waveinStop(wave_dev->sb_hw,
							       wave_in,
							       &dummy);
					wave_fmt = &wave_dev->wiinst->wave_fmt;
					wave_fmt->bitspersample = val;
					if (sblive_waveinQueryFormat(wave_dev->sb_hw->card_wavein,
								  wave_fmt,
								  CARDWAVE_QF_ALL) != CTSTATUS_SUCCESS){
						spin_unlock_irqrestore(&wiinst->lock, flags);
						return -EINVAL;
					}
					DPD("    set recording sample size -> ", wave_fmt->bitspersample);
					spin_unlock_irqrestore(&wiinst->lock, flags);
				}
				return put_user((val == 16) ? AFMT_S16_LE : AFMT_U8, (int *)arg);
			} else {
				struct wave_format *wave_fmt = NULL;

				if (file->f_mode & FMODE_READ)
					wave_fmt = &wave_dev->wiinst->wave_fmt;
				else if (file->f_mode & FMODE_WRITE)
					wave_fmt = &wave_dev->woinst->wave_fmt;
				return put_user((wave_fmt->bitspersample == 16) ? AFMT_S16_LE : AFMT_U8, (int *) arg);
			}
		}
		break;


		/* The following should not have to be implemented, but some older
		 * applications still use it. :(  The correct way to query the data
		 * format remains ioctl(fd, SNDCTL_DSP_SETFMT, &AFMT_QUERY); this is
		 * solely for backwards compatibility. [dave@pifke.org, 11/2/99]
		 */
	case SOUND_PCM_READ_BITS:
		{
			struct wave_format *wave_fmt = NULL;

			if (file->f_mode & FMODE_READ)
				wave_fmt = &wave_dev->wiinst->wave_fmt;
			else if ( file->f_mode & FMODE_WRITE )
				wave_fmt = &wave_dev->woinst->wave_fmt;
			return put_user((wave_fmt->bitspersample == 16) ? AFMT_S16_LE : AFMT_U8, (int *)arg);

		}
		break;

	case SNDCTL_DSP_GETTRIGGER:
		{
			int val = 0;

			PDEBUG("SNDCTL_DSP_GETTRIGGER:\n");
			if (file->f_mode & FMODE_WRITE && (wave_dev->enablebits & PCM_ENABLE_OUTPUT))
				val |= PCM_ENABLE_OUTPUT;
			if (file->f_mode & FMODE_READ && (wave_dev->enablebits & PCM_ENABLE_INPUT))
				val |= PCM_ENABLE_INPUT;
			return put_user(val, (int *) arg);
		}

	case SNDCTL_DSP_SETTRIGGER:
		{
			int val;
			u32 dummy;

			PDEBUG("SNDCTL_DSP_SETTRIGGER:\n");
			get_user_ret(val, (int *) arg, -EFAULT);
			if (file->f_mode & FMODE_WRITE) {
				struct woinst *woinst = wave_dev->woinst;
				struct wave_out *wave_out = woinst->wave_out;
				u32 flags;

				spin_lock_irqsave(&woinst->lock, flags);
				if (wave_out) {
					if (val & PCM_ENABLE_OUTPUT) {
						sblive_waveoutStart(wave_dev->sb_hw,
								    wave_out);
						wave_dev->enablebits |= PCM_ENABLE_OUTPUT;
					} else {
						sblive_waveoutStop(wave_dev->sb_hw,
								   wave_out,
								   &dummy);
						wave_dev->enablebits &= ~PCM_ENABLE_OUTPUT;
					}
				}
				spin_unlock_irqrestore(&woinst->lock, flags);
			}
			if (file->f_mode & FMODE_READ) {
				struct wiinst *wiinst = wave_dev->wiinst;
				struct wave_in *wave_in = wiinst->wave_in;
				u32 flags;

				spin_lock_irqsave(&wiinst->lock, flags);
				if (wave_in) {
					if (val & PCM_ENABLE_INPUT) {
						sblive_waveinStart(wave_dev->sb_hw,
								wave_in);
						wave_dev->enablebits |= PCM_ENABLE_INPUT;
					} else {
						sblive_waveinStop(wave_dev->sb_hw,
							       wave_in,
							       &dummy);
						wave_dev->enablebits &= ~PCM_ENABLE_INPUT;
					}
				}
				spin_unlock_irqrestore(&wiinst->lock, flags);
			}
			break;
		}

	case SNDCTL_DSP_GETOSPACE:
		{
			audio_buf_info info;
			PDEBUG("SNDCTL_DSP_GETOSPACE:\n");

			if (!(file->f_mode & FMODE_WRITE))
				return -EINVAL;
			{
				struct woinst *woinst = wave_dev->woinst;
				struct wave_out *wave_out = woinst->wave_out;
				u32 bytestocopy, pending, flags;

				spin_lock_irqsave(&woinst->lock, flags);
				if (wave_out) {
					if (sblive_waveoutGetXferSize(wave_dev->sb_hw->card_waveout,
								      wave_out,
								      &bytestocopy,
								      &pending)
					    != CTSTATUS_SUCCESS) {
						PDEBUG("sblive_waveoutGetXferSize failed\n");
					        spin_unlock_irqrestore(&woinst->lock, flags);
						return -EFAULT;
					}
					info.bytes = bytestocopy;
				} else {
					calculate_ofrag(woinst);
					info.bytes = woinst->numfrags * woinst->fragment_size;
				}

				if(woinst->silence_filled > 0){
					if(woinst->silence_filled == 1){
                        	        	if(pending <= woinst->fragment_size)
                                	        	woinst->silence_filled = 2;
                                		else
						info.bytes += woinst->fragment_size;
					}
					if(woinst->silence_filled == 2)
						info.bytes = woinst->numfrags * woinst->fragment_size;
				}

				info.fragstotal = woinst->numfrags;
				info.fragments = info.bytes / woinst->fragment_size;
				info.fragsize = woinst->fragment_size;
				spin_unlock_irqrestore(&woinst->lock, flags);
				if (copy_to_user((int *) arg, &info, sizeof(info)))
					return -EFAULT;
			}
		}
		break;

	case SNDCTL_DSP_GETISPACE:
		{
			audio_buf_info info;
			PDEBUG("SNDCTL_DSP_GETISPACE:\n");

			if (!(file->f_mode & FMODE_READ))
				return -EINVAL;
			{
				struct wiinst *wiinst = wave_dev->wiinst;
				struct wave_in *wave_in = wiinst->wave_in;
				u32 flags, bytestocopy, pending;

				spin_lock_irqsave(&wiinst->lock, flags);
				if (wave_in) {
					if (sblive_waveinGetXferSize(wave_dev->sb_hw->card_wavein,
								  wave_in,
								  &bytestocopy,
								  &pending) != CTSTATUS_SUCCESS) {
						PDEBUG("sblive_waveinGetXferSize failed\n");
					        spin_unlock_irqrestore(&wiinst->lock, flags);
						return -EFAULT;
					}
					info.bytes = bytestocopy;
				} else {
					calculate_ifrag(wiinst);
					info.bytes = wiinst->numfrags * wiinst->fragment_size;
				}
				info.fragstotal = wiinst->numfrags;
				info.fragments = info.bytes / wiinst->fragment_size;
				info.fragsize = wiinst->fragment_size;
				spin_unlock_irqrestore(&wiinst->lock, flags);
				if (copy_to_user((int *) arg, &info, sizeof(info)))
					return -EFAULT;
			}
		}
		break;

	case SNDCTL_DSP_NONBLOCK:
		PDEBUG("SNDCTL_DSP_NONBLOCK:\n");
		file->f_flags |= O_NONBLOCK;
		break;

	case SNDCTL_DSP_GETODELAY:
		{
			u32 flags, pending, bytestocopy;
			struct woinst *woinst = wave_dev->woinst;
			struct wave_out *wave_out = woinst->wave_out;
			int val = 0;
			PDEBUG("SNDCTL_DSP_GETODELAY:\n");
			if (!(file->f_mode & FMODE_WRITE))
				return -EINVAL;

			spin_lock_irqsave(&woinst->lock, flags);
			if (wave_out) {
				if (sblive_waveoutGetXferSize(wave_dev->sb_hw->card_waveout,
							   wave_out,
							   &bytestocopy,
							   &pending) != CTSTATUS_SUCCESS) {
					PDEBUG("sblive_waveoutGetXferSize failed\n");
					spin_unlock_irqrestore(&woinst->lock, flags);
					return -EFAULT;
				}
				
				if(woinst->silence_filled > 0){
					if(woinst->silence_filled == 1){
                                        	if(pending <= woinst->fragment_size)
                                                	woinst->silence_filled = 2;
                                        	else
							pending -= woinst->fragment_size;
					}
					if(woinst->silence_filled == 2)
						pending = 0;
                                }
				val = pending;
			}
			spin_unlock_irqrestore(&woinst->lock, flags);
			return put_user(val, (int *) arg);
		}

	case SNDCTL_DSP_GETIPTR:
		PDEBUG("SNDCTL_DSP_GETIPTR: not implemented\n");
		break;

	case SNDCTL_DSP_GETOPTR:
		{
			count_info cinfo;
			u32 flags;
			struct woinst *woinst = wave_dev->woinst;
			struct wave_out *wave_out = woinst->wave_out;

			PDEBUG("SNDCTL_DSP_GETOPTR:\n");
			if (!(file->f_mode & FMODE_WRITE))
				return -EINVAL;

			spin_lock_irqsave(&woinst->lock, flags);
			if (wave_out) {
				if (sblive_waveoutGetControl(wave_dev->sb_hw->card_waveout,
							     wave_out,
							     WAVECURPOS,
							     (u32 *) &cinfo.ptr)
				    != CTSTATUS_SUCCESS) {
					spin_unlock_irqrestore(&woinst->lock, flags);
					return -EINVAL;
				}
			} else
				cinfo.ptr = 0;
			
			DPD(" cinfo.ptr -> ", cinfo.ptr);
			cinfo.bytes = cinfo.ptr + woinst->total_played - woinst->total_played % (woinst->fragment_size * woinst->numfrags);

			cinfo.blocks = cinfo.bytes / woinst->fragment_size - woinst->getoptr_blocks;

			woinst->getoptr_blocks = cinfo.bytes / woinst->fragment_size;

			spin_unlock_irqrestore(&woinst->lock, flags);
			return copy_to_user((void *) arg, &cinfo, sizeof(cinfo));
		}

	case SNDCTL_DSP_GETBLKSIZE:
		PDEBUG("SNDCTL_DSP_GETBLKSIZE:\n");
		if (file->f_mode & FMODE_WRITE) {
			struct woinst *woinst = wave_dev->woinst;
			u32 flags;

			spin_lock_irqsave(&woinst->lock, flags);
			calculate_ofrag(wave_dev->woinst);
			spin_unlock_irqrestore(&woinst->lock, flags);
			return put_user(wave_dev->woinst->fragment_size, (int *) arg);
		}
		if (file->f_mode & FMODE_READ) {
			struct wiinst *wiinst = wave_dev->wiinst;
                        u32 flags;

			spin_lock_irqsave(&wiinst->lock, flags);
			calculate_ifrag(wave_dev->wiinst);
			spin_unlock_irqrestore(&wiinst->lock, flags);
			return put_user(wave_dev->wiinst->fragment_size, (int *) arg);
		}
		break;
	case SNDCTL_DSP_POST:
		PDEBUG("SNDCTL_DSP_POST: not implemented\n");
		break;

	case SNDCTL_DSP_SUBDIVIDE:
		PDEBUG("SNDCTL_DSP_SUBDIVIDE: not implemented\n");
		break;

	case SNDCTL_DSP_SETFRAGMENT:
		{
			int val;
			PDEBUG("SNDCTL_DSP_SETFRAGMENT:\n");
			get_user_ret(val, (int *) arg, -EFAULT);

			DPD(" val is ", val);

			if (val == 0)
				return -EIO;

			if (file->f_mode & FMODE_WRITE) {
				struct woinst *woinst = wave_dev->woinst;
				struct wave_out *wave_out = woinst->wave_out;

				if (wave_out)
					return -EINVAL;		// too late to change

				woinst->ossfragshift = val & 0xffff;
				woinst->numfrags = (val >> 16) & 0xffff;
			}

			if (file->f_mode & FMODE_READ) {
				struct wiinst *wiinst = wave_dev->wiinst;
				struct wave_in *wave_in = wiinst->wave_in;

				if (wave_in)
					return -EINVAL;		// too late to change

				wiinst->ossfragshift = val & 0xffff;
				wiinst->numfrags = (val >> 16) & 0xffff;
			}

			break;
		}
	default:		// Default is unrecognized command
		PDEBUG("default: %x \n", cmd);
		return -EINVAL;
	}
	return 0;
}

#if LINUX_VERSION_CODE < 0x020100
static int emu10k1_audio_mmap(struct inode *inode, struct file *file, struct vm_area_struct *vma)
#else
static int emu10k1_audio_mmap(struct file *file, struct vm_area_struct *vma)
#endif
{
	struct sblive_wavedevice *wave_dev = (struct sblive_wavedevice *) file->private_data;
	struct woinst *woinst = wave_dev->woinst;
	u32 size, flags;

	PDEBUG("emu10k1_audio_mmap() called\n");

	if (!(vma->vm_flags & VM_WRITE))
		return -EINVAL;

#if LINUX_VERSION_CODE < 0x020319
	if (vma->vm_offset != 0)
#else
	if (vma->vm_pgoff != 0)
#endif /* LINUX_VERSION_CODE < 0x020319 */
		return -ENXIO;

	spin_lock_irqsave(&woinst->lock, flags);
	if (!woinst->wave_out) {
		calculate_ofrag(woinst);
		if (sblive_waveoutOpen(wave_dev->sb_hw,
				       &woinst->wave_fmt,
				       waveOutCallbackFn,
				       (u32) wave_dev,
				       &woinst->fragment_size,
				       woinst->numfrags,
				       &woinst->wave_out)
		    != CTSTATUS_SUCCESS) {
			DPF("    sblive_waveoutOpen failed!");
			spin_unlock_irqrestore(&woinst->lock, flags);
			return -EINVAL;
		}
		/* now mark the pages as reserved, otherwise remap_page_range doesn't do what we want */
		if (woinst->wave_out->memhandle) {
			u32 order = ((PLMEMHANDLE) woinst->wave_out->memhandle)->order;
			u32 map, mapend;

			mapend = MAP_NR(woinst->wave_out->wavexferbuf->xferbuffer + (PAGE_SIZE << order) - 1);
			for (map = MAP_NR(woinst->wave_out->wavexferbuf->xferbuffer); map <= mapend; map++)
				set_bit(PG_reserved, &mem_map[map].flags);
		}
	}
	size = vma->vm_end - vma->vm_start;

	if (size > (PAGE_SIZE << ((PLMEMHANDLE) woinst->wave_out->memhandle)->order))
		return -EINVAL;
	// In Linux, get_free_pages returns a virtual address (it may also be
	// physical but nothing is guaranteed. We have to do virt_to_phys here.
	if (remap_page_range(vma->vm_start, virt_to_phys(((PLMEMHANDLE) woinst->wave_out->memhandle)->virtaddx), size, vma->vm_page_prot)){
		spin_unlock_irqrestore(&woinst->lock, flags);
		return -EAGAIN;
	}

	woinst->mapped = TRUE;

#if LINUX_VERSION_CODE < 0x020100
	vma->vm_inode = inode;
	inode->i_count++;
#else
	/* In Linux 2.2.4 or higher this is handled nicely by the VM for us */
	//vma->vm_file = file;
	//file->f_count++;
#endif
	spin_unlock_irqrestore(&woinst->lock, flags);
	return 0;
}

static int emu10k1_audio_open(struct inode *inode, struct file *file)
{
	int minor = MINOR(inode->i_rdev);
	struct sblive_hw * sb_hw = sblive_devs;
	struct sblive_wavedevice * wave_dev;

	PDEBUG("emu10k1_audio_open() called \n");

	/* check for correct device to open */
	while (sb_hw && ((sb_hw->audio_num ^ minor) & ~0xf))
		sb_hw = sb_hw->next;
	if (!sb_hw)
		return -ENODEV;

	/* wait for device to become free */
	down(&sb_hw->open_sem);
	while ((file->f_mode & FMODE_READ) && (sb_hw->open_mode & FMODE_READ)) {
		if (file->f_flags & O_NONBLOCK) {
			up(&sb_hw->open_sem);
			return -EAGAIN;
		}
		up(&sb_hw->open_sem);
		DPF("  open sleep ...");
		interruptible_sleep_on(&sb_hw->open_wait);
		if (signal_pending(current))
			return -ERESTARTSYS;
		down(&sb_hw->open_sem);
	}

	wave_dev = (struct sblive_wavedevice *)kmalloc(sizeof(struct sblive_wavedevice), GFP_KERNEL);
	if (!wave_dev) {
		DPF(" struct sblive_wavedevice alloc fail.");
		up(&sb_hw->open_sem);
		return -EINVAL;
	}
#ifdef MEMTEST
	PDEBUG("\ncardwo.c: kmalloc: [%p]\n", wave_dev);
#endif
	wave_dev->sb_hw = sb_hw;
	wave_dev->wiinst = NULL;
	wave_dev->woinst = NULL;
	wave_dev->enablebits = PCM_ENABLE_OUTPUT | PCM_ENABLE_INPUT;	//default

	if (file->f_mode & FMODE_WRITE) {
		struct woinst *woinst;

		woinst = (struct woinst *) kmalloc(sizeof(struct woinst), GFP_KERNEL);
		if (!woinst) {
			DPF("struct woinst alloc failed.");
			up(&sb_hw->open_sem);
			return -ENODEV;
		}
#ifdef MEMTEST
		PDEBUG("\naudio.c: kmalloc: [%p]\n", woinst);
#endif
		// set default format to : mono 8-bit 8kHz
		woinst->wave_fmt.flags = 0;
		woinst->wave_fmt.samplingrate = 8000;
		woinst->wave_fmt.bitspersample = 8;
		woinst->wave_fmt.channels = 1;
		woinst->wave_fmt.isinput = (file->f_mode == FMODE_READ) ? TRUE : FALSE;
		woinst->ossfragshift = 0;
		woinst->fragment_size = 0;
		woinst->numfrags = 0;
		woinst->wave_out = NULL;	// wave_out instance

#if LINUX_VERSION_CODE < 0x020301
		init_waitqueue(&woinst->wait_queue);
#else
		init_waitqueue_head(&woinst->wait_queue);
#endif /* LINUX_VERSION_CODE < 0x020301 */
		woinst->mapped = FALSE;
		woinst->total_copied = 0;
		woinst->total_played = 0;
		woinst->silence_filled = 0;
		woinst->silence_start = 0;
		woinst->getoptr_blocks = 0;
		woinst->wave_ptr = 0;
		woinst->lock = SPIN_LOCK_UNLOCKED;
		wave_dev->woinst = woinst;
	}
	if (file->f_mode & FMODE_READ) {
		/* recording */
		struct wiinst *wiinst;

		/* only support one wave input instance */
		if (sb_hw->card_wavein->numrecordinst >= 1) {
			DPF("  exceed one wave input instance.");
			up(&sb_hw->open_sem);
			return -ENODEV;
		}
		wiinst = (struct wiinst *) kmalloc(sizeof(struct wiinst), GFP_KERNEL);
		if (!wiinst) {
			DPF("struct wiinst alloc failed.");
			up(&sb_hw->open_sem);
			return -ENODEV;
		}
#ifdef MEMTEST
		PDEBUG("\ncardwo.c: kmalloc: [%p]\n", wiinst);
#endif
		// set default format to : mono 8-bit 8kHz
		wiinst->wave_fmt.flags = 0;
		wiinst->wave_fmt.samplingrate = 8000;
		wiinst->wave_fmt.bitspersample = 8;
		wiinst->wave_fmt.channels = 1;
		wiinst->wave_fmt.isinput = (file->f_mode == FMODE_READ) ? TRUE : FALSE;
		wiinst->ossfragshift = 0;
		wiinst->fragment_size = 0;
		wiinst->numfrags = 0;
		wiinst->wave_in = NULL;	// sblive_wavein instance
		wiinst->lock = SPIN_LOCK_UNLOCKED;
		wave_dev->wiinst = wiinst;
		wave_dev->sb_hw->card_wavein->numrecordinst++;
#ifdef EMU10K1_DEBUG
		if (wave_dev->woinst != NULL)
			DPF(" audio_open opened both WAVEOUT and WAVEIN!");
#endif
	}
	file->private_data = (void *) wave_dev;

	sb_hw->open_mode |= file->f_mode & FMODE_READ;
	up(&sb_hw->open_sem);

	MOD_INC_USE_COUNT;
	return 0;		// Success?

}

#if LINUX_VERSION_CODE < 0x020100
static void emu10k1_audio_release(struct inode *inode, struct file *file)
#else
static int emu10k1_audio_release(struct inode *inode, struct file *file)
#endif
{
	struct sblive_wavedevice * wave_dev = (struct sblive_wavedevice *) file->private_data;
	struct sblive_hw * sb_hw = wave_dev->sb_hw;
	PDEBUG("emu10k1_audio_release() called\n");

	if (file->f_mode & FMODE_WRITE) {
		struct woinst *woinst = wave_dev->woinst;
		struct wave_out *wave_out = woinst->wave_out;

		if (wave_out) {
			if ((wave_out->state == CARDWAVE_STATE_STARTED) && !(file->f_flags & O_NONBLOCK)) {
				while (!signal_pending(current)) {
					if (woinst->total_played < woinst->total_copied) {
						DPF(" Buffer haven't been totally played, sleep ...");
						interruptible_sleep_on(&woinst->wait_queue);
					} else
						break;
				}
			}
			if (woinst->mapped && wave_out->memhandle) {
				u32 order = ((PLMEMHANDLE) wave_out->memhandle)->order;
				u32 map, mapend;

				/* undo marking the pages as reserved */
				mapend = MAP_NR(wave_out->wavexferbuf->xferbuffer + (PAGE_SIZE << order) - 1);
				for (map = MAP_NR(wave_out->wavexferbuf->xferbuffer); map <= mapend; map++)
					clear_bit(PG_reserved, &mem_map[map].flags);
			}
			woinst->mapped = FALSE;
			sblive_waveoutClose(wave_dev->sb_hw, wave_out);
		}
		kfree((void *) wave_dev->woinst);
	}
	if (file->f_mode & FMODE_READ) {
		struct wave_in *wave_in = wave_dev->wiinst->wave_in;

		if (wave_in)
			sblive_waveinClose(wave_dev->sb_hw, wave_in);
		kfree((void *) wave_dev->wiinst);
#ifdef MEMTEST
		PDEBUG("\ncardwo.c: kfree: [%p]\n", wave_dev->wiinst);
#endif
		wave_dev->sb_hw->card_wavein->numrecordinst--;
	}
	kfree((void *) wave_dev);
#ifdef MEMTEST
	PDEBUG("\ncardwo.c: kfree: [%p]\n", wave_dev);
#endif

	down(&sb_hw->open_sem);
	sb_hw->open_mode &= ~(file->f_mode & FMODE_READ);
	up(&sb_hw->open_sem);
	wake_up(&sb_hw->open_wait);
	MOD_DEC_USE_COUNT;

#if LINUX_VERSION_CODE >= 0x020100
	return 0;
#endif
}


#if LINUX_VERSION_CODE < 0x020100
static int emu10k1_audio_poll(struct inode *inode, struct file *file, int sel_type, select_table * wait)
{
	struct sblive_wavedevice * wave_dev = (struct sblive_wavedevice *) file->private_data;
	u32 flags, bytestocopy, pending;

	PDEBUG("emu10k1_audio_select called.\n");
	if (sel_type == SEL_OUT && file->f_mode & FMODE_WRITE) {
		spin_lock_irqsave(&wave_dev->woinst->lock, flags);
		struct wave_out *wave_out = wave_dev->woinst->wave_out;

		if (wave_out) {
			if (sblive_waveoutGetXferSize(wave_dev->sb_hw->card_waveout,
						      (u32) wave_out,
						      &bytestocopy,
						      &pending)
			    == CTSTATUS_SUCCESS) {
				if (bytestocopy > 0) {
					spin_unlock_irqrestore(&wave_dev->woinst->lock, flags);
					return 1;	// there is space in the playback buffer
				}
				DPD("    bytestocopy --> ", bytestocopy);
				select_wait(&wave_dev->woinst->wait_queue, wait);
			}
		}
		spin_unlock_irqrestore(&wave_dev->woinst->lock, flags);
	}
	if (sel_type == SEL_IN && file->f_mode & FMODE_READ) {
		spin_lock_irqsave(&wave_dev->wiinst->lock, flags);
		struct wave_in *wave_in = wave_dev->wiinst->wave_in;

		if (sblive_waveinGetXferSize(wave_dev->sb_hw->card_wavein,
					  wave_in,
					  &bytestocopy,
					  &pending) == CTSTATUS_SUCCESS) {
			if (bytestocopy > 0) {
				spin_unlock_irqrestore(&wave_dev->wiinst->lock, flags);
				return 1;
			}
			DPD("    bytestocopy --> ", bytestocopy);
			select_wait(&wave_in->wait_queue, wait);
		}
		spin_unlock_irqrestore(&wave_dev->wiinst->lock, flags);
	}
	return 0;
}

#else
static unsigned int emu10k1_audio_poll(struct file *file, struct poll_table_struct *wait)
{
	struct sblive_wavedevice *wave_dev = (struct sblive_wavedevice *) file->private_data;
	u32 flags, bytestocopy, pending;
	u16 mask = 0;

	PDEBUG("emu10k1_audio_poll() called\n");

	if ((file->f_mode & FMODE_READ) && wave_dev->wiinst->wave_in)
		poll_wait(file, &wave_dev->wiinst->wave_in->wait_queue, wait);
	if ((file->f_mode & FMODE_WRITE) && wave_dev->woinst)
		poll_wait(file, &wave_dev->woinst->wait_queue, wait);

	if (file->f_mode & FMODE_READ) {
		struct wave_in *wave_in = wave_dev->wiinst->wave_in;

		if (!wave_in)
			mask |= POLLIN | POLLRDNORM;
		else {
			int status;

			spin_lock_irqsave(&wave_dev->wiinst->lock, flags);
                        status = sblive_waveinGetXferSize(wave_dev->sb_hw->card_wavein,
						       wave_in,
						       &bytestocopy,
						       &pending);
			spin_unlock_irqrestore(&wave_dev->wiinst->lock, flags);
			if (status == CTSTATUS_SUCCESS) {
				if (bytestocopy > 0)
					mask |= POLLIN | POLLRDNORM;
			}
		}
	}
	if (file->f_mode & FMODE_WRITE)
	{
		struct wave_out *wave_out = wave_dev->woinst->wave_out;

		if (!wave_out)
			mask |= POLLOUT | POLLWRNORM;
		else {
			int status;
			spin_lock_irqsave(&wave_dev->woinst->lock, flags);
			status = sblive_waveoutGetXferSize(wave_dev->sb_hw->card_waveout,
							   wave_out,
							   &bytestocopy,
							   &pending);
			spin_unlock_irqrestore(&wave_dev->woinst->lock, flags);
			if (status == CTSTATUS_SUCCESS) {
				if (bytestocopy > 0)
					mask |= POLLOUT | POLLWRNORM;
			}
		}
	}

	return mask;
}

#endif				// LINUX_VERSION_CODE



static int waveOutCallbackFn(unsigned long message, unsigned long refdata, unsigned long param)
{
	struct sblive_wavedevice *wave_dev = (struct sblive_wavedevice *) refdata;
	struct woinst *woinst = wave_dev->woinst;
	struct wave_out *wave_out = woinst->wave_out;
	unsigned long flags;

	spin_lock_irqsave(&woinst->lock, flags);
	switch (message) {
		
	case CARDWAVE_EVENT_TIMER:
		{
			u32 bytestocopy, pending, waveptr;

			//DPD(" IRQ! for wave Instance ->  ", wave_out);

			if (sblive_waveoutGetControl(wave_dev->sb_hw->card_waveout,
						     wave_out,
						     WAVECURPOS,
						     (u32 *) & waveptr)
			    == CTSTATUS_SUCCESS) {
				woinst->total_played += waveptr - woinst->wave_ptr;

                                if(waveptr < woinst->wave_ptr)
                        		woinst->total_played += woinst->fragment_size * woinst->numfrags;

                                 woinst->wave_ptr = waveptr;
			}

			if (woinst->mapped)
				break;

			if (sblive_waveoutGetXferSize(wave_dev->sb_hw->card_waveout,
						      wave_out,
						      &bytestocopy,
						      &pending)
			    == CTSTATUS_SUCCESS) {
				if (bytestocopy >= woinst->fragment_size || woinst->silence_filled > 0) {
					if (pending <= woinst->fragment_size && bytestocopy >= woinst->fragment_size) {
						if(woinst->silence_filled == 0)
							sblive_waveoutGetControl(wave_dev->sb_hw->card_waveout,
									      wave_out,
									      WAVEWRITEPOINTER,
								   	      &woinst->silence_start);

						bytestocopy = woinst->fragment_size;
						sblive_waveoutFillSilence(wave_dev->sb_hw->card_waveout,
								       wave_out,
								       &bytestocopy);
						if(woinst->silence_filled < 2)
							woinst->silence_filled++;
					}
					wake_up(&woinst->wait_queue);
				} else {
					DPD("  Not enough transfer size, ", bytestocopy);
				}
			}
		}
		break;

	default:
		break;

	}
	spin_unlock_irqrestore(&woinst->lock, flags);
	return CTSTATUS_SUCCESS;
}


static int waveInCallbackFn(unsigned long message, unsigned long refdata, unsigned long param)
{
	struct sblive_wavedevice *wave_dev = (struct sblive_wavedevice *) refdata;
	struct wiinst *wiinst = wave_dev->wiinst;
	struct wave_in *wave_in = wiinst->wave_in;
	unsigned long flags;

        /* Are these locks necessary around the wake_up ? kabi@i.am */
	spin_lock_irqsave(&wiinst->lock, flags);
	switch (message) {
	case CARDWAVE_EVENT_CALLBACKSIZEREACHED:
		{
			u32 bytestocopy, pending;

			DPF(" IRQ! ");
			if (sblive_waveinGetXferSize(wave_dev->sb_hw->card_wavein,
						  wave_in,
						  &bytestocopy,
						  &pending) == CTSTATUS_SUCCESS) {

				if (bytestocopy >= wiinst->fragment_size) {
					DPF(" Wake UP!");
					wake_up(&wave_in->wait_queue);
				} else {
					DPD("  Not enough transfer size, ", bytestocopy);
				}
			}
		}
		break;

	default:
		break;
	}
	spin_unlock_irqrestore(&wiinst->lock, flags);
	return CTSTATUS_SUCCESS;
}

static void calculate_ofrag(struct woinst *woinst)
{
	u32 fragsize = 0, bytespersec;
	if (woinst->fragment_size) {
		DPF(" fragment_size already calculated!");
		return;
	}
	bytespersec = woinst->wave_fmt.channels
		* (woinst->wave_fmt.bitspersample >> 3)
		* woinst->wave_fmt.samplingrate;

	woinst->fragment_size = 1;

	if (woinst->ossfragshift) {
		woinst->fragment_size <<= (woinst->ossfragshift < MINFRAGSHIFT ? MINFRAGSHIFT : woinst->ossfragshift);
	} else {
		fragsize = (bytespersec * WAVEOUT_DEFAULTFRAGLEN) / 1000 - 1;
	}

	while (fragsize) {
		fragsize >>= 1;
		woinst->fragment_size <<= 1;
	}

	if (!woinst->numfrags){
		u32 numfrags;

		numfrags = (bytespersec * WAVEOUT_DEFAULTBUFLEN) 
			/ (woinst->fragment_size * 1000) - 1;

		woinst->numfrags = 1;
		while (numfrags) {
                	numfrags >>= 1;
                	woinst->numfrags <<= 1;
		}
	}

	if (woinst->numfrags < MINFRAGS) woinst->numfrags = MINFRAGS;

	if (woinst->numfrags * woinst->fragment_size > WAVEOUT_MAXBUFSIZE) {
		woinst->numfrags = WAVEOUT_MAXBUFSIZE / woinst->fragment_size;
		if (woinst->numfrags < MINFRAGS) {
			woinst->numfrags = MINFRAGS;
			woinst->fragment_size = WAVEOUT_MAXBUFSIZE / MINFRAGS;
		}
	} else if (woinst->numfrags * woinst->fragment_size < WAVEOUT_MINBUFSIZE) {
		woinst->numfrags = WAVEOUT_MINBUFSIZE / woinst->fragment_size;
	}	
	
	DPD(" calculated playback fragment_size -> ", woinst->fragment_size);
	DPD(" calculated playback numfrags -> ",woinst->numfrags);
}

static void calculate_ifrag(struct wiinst *wiinst)
{
	u32 fragsize, bytespersec;
	if (wiinst->fragment_size) {
		DPF(" fragment_size already calculated!");
		return;
	}
	bytespersec = wiinst->wave_fmt.channels
		* (wiinst->wave_fmt.bitspersample >> 3)
		* wiinst->wave_fmt.samplingrate;

	wiinst->fragment_size = 1;

	if (wiinst->ossfragshift)
		wiinst->fragment_size <<= (wiinst->ossfragshift < MINFRAGSHIFT ? MINFRAGSHIFT : wiinst->ossfragshift);
	else {
		fragsize = (bytespersec * WAVEIN_DEFAULTFRAGLEN) / 1000 - 1;
		while (fragsize) {
			fragsize >>= 1;
			wiinst->fragment_size <<= 1;
		}
	}

	if (!wiinst->numfrags)
		wiinst->numfrags = (WAVEIN_DEFAULTBUFLEN * bytespersec)
			/ (wiinst->fragment_size * 1000);

	if (wiinst->numfrags < MINFRAGS) wiinst->numfrags = MINFRAGS;

	if (wiinst->numfrags * wiinst->fragment_size > WAVEIN_MAXBUFSIZE) {
		wiinst->numfrags = WAVEIN_MAXBUFSIZE / wiinst->fragment_size;
		if (wiinst->numfrags < MINFRAGS) {
			wiinst->numfrags = MINFRAGS;
			wiinst->fragment_size = WAVEIN_MAXBUFSIZE / MINFRAGS;
		}
	}

	DPD(" calculated recording fragment_size -> ", wiinst->fragment_size);
	DPD(" calculated recording numfrags -> ",wiinst->numfrags);
}

int audio_init(struct sblive_hw * sb_hw, u8 * pname)
{
	// Initialize CardwaveOut struct
	sb_hw->card_waveout = kmalloc(sizeof(struct sblive_waveout), GFP_KERNEL);
	if (!sb_hw->card_waveout) {
		printk(KERN_WARNING "sblive: Unable to allocate struct sblive_waveout: out of memory\n");
		return CTSTATUS_ERROR;
	}
#ifdef MEMTEST
	PDEBUG("\naudio.c: kmalloc: [%p]\n", sb_hw->card_waveout);
#endif
	memset(sb_hw->card_waveout, 0, sizeof(struct sblive_waveout));

	sblive_waveoutInit(sb_hw->card_waveout, pname);

	// Initialize CardwaveIn struct
	sb_hw->card_wavein = kmalloc(sizeof(struct sblive_wavein), GFP_KERNEL);
	if (!sb_hw->card_wavein) {
		kfree(sb_hw->card_waveout);
		sb_hw->card_waveout = NULL;
		printk(KERN_WARNING "sblive: Unable to allocate struct sblive_wavein: out of memory\n");
		return CTSTATUS_ERROR;
	}
#ifdef MEMTEST
	PDEBUG("\naudio.c: kmalloc: [%p]\n", sb_hw->card_wavein);
#endif
	memset(sb_hw->card_wavein, 0, sizeof(struct sblive_wavein));

	sblive_waveinInit(sb_hw->card_wavein, pname);

	return CTSTATUS_SUCCESS;
}

int audio_exit(struct sblive_hw *sb_hw)
{
	sblive_waveoutExit(sb_hw->card_waveout);
	sblive_waveinExit(sb_hw->card_wavein);
	kfree(sb_hw->card_waveout);
	kfree(sb_hw->card_wavein);
#ifdef MEMTEST
	PDEBUG("\naudio.c: kfree: [%p]\n", sb_hw->card_waveout);
	PDEBUG("\naudio.c: kfree: [%p]\n", sb_hw->card_wavein);
#endif
	sb_hw->card_waveout = NULL;
	sb_hw->card_wavein = NULL;

	return CTSTATUS_SUCCESS;
}


struct file_operations emu10k1_audio_fops =
{
	&emu10k1_audio_llseek,
	&emu10k1_audio_read,
	&emu10k1_audio_write,
	NULL,
	&emu10k1_audio_poll,
	&emu10k1_audio_ioctl,
	&emu10k1_audio_mmap,
	&emu10k1_audio_open,
#if LINUX_VERSION_CODE >= 0x020100
	NULL,			/* flush */
#endif
	&emu10k1_audio_release,
	NULL,
	NULL,
	NULL
};
