/*
 **********************************************************************
 *     hwaccess.c -- Hardware access layer
 *     Copyright 1999, 2000 Creative Labs, Inc.
 *
 **********************************************************************
 *
 *     Date                 Author          Summary of changes
 *     ----                 ------          ------------------
 *     October 20, 1999     Bertrand Lee    base code release
 *
 **********************************************************************
 *
 *     This program is free software; you can redistribute it and/or
 *     modify it under the terms of the GNU General Public License as
 *     published by the Free Software Foundation; either version 2 of
 *     the License, or (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public
 *     License along with this program; if not, write to the Free
 *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 *     USA.
 *
 **********************************************************************
 */

#include "hwaccess.h"
#include "mycommon.h"

#if LINUX_VERSION_CODE < 0x020100
#include <linux/delay.h>
#endif
  

spinlock_t sblive_spinlock = SPIN_LOCK_UNLOCKED;

int halHWInit(struct sblive_hw * sb_hw);


/*
 *	halInit(struct sblive_hw *, struct sblive_config *, u32, u32 *)
 */

int halInit(struct sblive_hw * sb_hw, struct sblive_config * config,
		   u32 oshandle, u32 * hwflags)
{
	unsigned long ioaddr;

	/* should fill in default values here */
	sb_hw->paneffectsbus = 0;
	sb_hw->auxeffectsbus = 1;
	sb_hw->choruseffectsbus = 2;
	sb_hw->reverbeffectsbus = 3;

	// Setup Critical Sections
	spin_lock_init(&sb_hw->emu_lock);

	sb_hw->numvoices = NUM_G;
	sb_hw->oshandle = oshandle;
	sb_hw->dsCardCfg = *config;

	sb_hw->hwaddr = config->ioportbase[0];
	ioaddr = config->ioportbase[0];

	sb_hw->synthaddx = (u32) ioaddr;
	sb_hw->mpuaddx = (u32) (ioaddr + MUDATA);
	sb_hw->mixeraddx = (u32) (ioaddr + AC97D);
	sb_hw->intrenabaddx = (u32) (ioaddr + INTE);
	sb_hw->intrStatusAddx = (u32) (ioaddr + IPR);
	sb_hw->wallclockaddx = (u32) (ioaddr + WC);
	sb_hw->hwconfigaddx = (u32) (ioaddr + HCFG);
	sb_hw->timeraddx = (u32) (ioaddr + TIMR);

	*hwflags = 0;
	*hwflags |= HWFLAG_EMU;
	*hwflags |= HWFLAG_MPU;
	*hwflags |= HWFLAG_MIXER;
	*hwflags |= HWFLAG_EFFECT;


	sb_hw->irq = config->IRQregs[0];
	if (sb_hw->irq) {
		if (sb_hw->irq == 2)
			sb_hw->irq = 9;
		*hwflags |= HWFLAG_IRQ;
	}
	// Init Card
	if (halHWInit(sb_hw) != CTSTATUS_SUCCESS)
		return CTSTATUS_ERROR;

	SETMAGIC(sb_hw);

	if (sblive_irqmgrInit(sb_hw) != CTSTATUS_SUCCESS) {
		ASSERT(FALSE);
		*hwflags &= ~HWFLAG_IRQ;
		sb_hw->irq = 0;
	}
	sblive_voiceInit(sb_hw);
	sblive_timerInit(sb_hw);
	emu10kaddxmgrInit(sb_hw);


	DPD("  hw control register -> ", inl(sb_hw->hwconfigaddx));
	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//*  halExit(struct sblive_hw * sb_hw)                                        *
//*****************************************************************************
int halExit(struct sblive_hw * sb_hw)
{
	int ch;

	TESTMAGIC(sb_hw);

	sblive_voiceExit(sb_hw);
	sblive_timerExit(sb_hw);
	emu10kaddxmgrExit(sb_hw);

	if (sb_hw->irq) {
		outl(0,sb_hw->intrenabaddx);
		sblive_irqmgrExit(sb_hw);
	}

	/** shutdown the chip **/

	for (ch = 0; ch < NUM_G; ch++)
		sblive_writesynth(sb_hw, DCYSUSV | ch, ENV_OFF);
	for (ch = 0; ch < NUM_G; ch++) {
		sblive_writesynth(sb_hw, VTFT | ch, 0);
		sblive_writesynth(sb_hw, CVCF | ch, 0);
		sblive_writesynth(sb_hw, PTRX | ch, 0);
		sblive_writesynth(sb_hw, CPF | ch, 0);
	}

	/* reset recording buffers */
	sblive_writesynth(sb_hw, MICBS, 0);
	sblive_writesynth(sb_hw, MICBA, 0);
	sblive_writesynth(sb_hw, FXBS, 0);
	sblive_writesynth(sb_hw, FXBA, 0);
	sblive_writesynth(sb_hw, FXWC, 0);
	sblive_writesynth(sb_hw, ADCBS, 0);
	sblive_writesynth(sb_hw, ADCBA, 0);
	sblive_writesynth(sb_hw, TMBS, 0);
	sblive_writesynth(sb_hw, TMBA, 0);
	sblive_writesynth(sb_hw, DBG, 0x8000);

	/* disable channel interrupt */
	sblive_writesynth(sb_hw, CLIEL, 0);
	sblive_writesynth(sb_hw, CLIEH, 0);
	sblive_writesynth(sb_hw, SOLEL, 0);
	sblive_writesynth(sb_hw, SOLEH, 0);

	/* disable audio and lock cache */
	outl(0x0e, sb_hw->hwconfigaddx);
	sblive_writesynth(sb_hw, PTB, 0);

#ifdef TANKMEM
	free_pages(sb_hw->tankmem, sb_hw->tmemsizeIdx + 2);
#endif				// TANKMEM
	free_pages(sb_hw->virtualpagetable, 4);
#ifdef MEMTEST
	PDEBUG("\nhwaccess.c: free_pages: [%x]\n", sb_hw->virtualpagetable);
#endif

	osFreeMemPhysical(sb_hw->silentpage);

	CLEARMAGIC(sb_hw);

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//*  halSetPowerState (struct sblive_hw * sb_hw, u32 powerstate)              *
//*****************************************************************************
int halSetPowerState(struct sblive_hw * sb_hw, u32 powerstate)
{
	u32 hwconf;

	sb_hw->powerstate = powerstate;
	switch (powerstate) {
	case POWERSTATE_STANDBY:
	case POWERSTATE_SUSPEND:
		// disable all interrupts
		sb_hw->intrstate = inl(sb_hw->intrenabaddx);
		outl(0, sb_hw->intrenabaddx);
		// digital mute
		hwconf = inl(sb_hw->hwconfigaddx) & ~0x1f;
		outl(hwconf + 16 + 2, sb_hw->hwconfigaddx);
		break;

	case POWERSTATE_STANDBYRESUME:
	case POWERSTATE_SUSPENDRESUME:
		// unmute
		halWC_WAIT(sb_hw, 1024);
		hwconf = inl(sb_hw->hwconfigaddx) & ~0x1f;
		outl(1 + 16 + hwconf, sb_hw->hwconfigaddx);
		// re-enable interrupts
		outl(sb_hw->intrstate,  sb_hw->intrenabaddx);
		break;

	default:
		break;
	}

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//*  halHWInit (struct sblive_hw * sb_hw)                                     *
//*****************************************************************************
int halHWInit(struct sblive_hw * sb_hw)
{
	int nCh;
	u32 size = 0;
	u32 sizeIdx = 0;
	int status;

	/* disable audio and lock cache */
	outl(0x0e, sb_hw->hwconfigaddx);

	/* reset recording buffers */
	sblive_writesynth(sb_hw, MICBS, 0);
	sblive_writesynth(sb_hw, MICBA, 0);
	sblive_writesynth(sb_hw, FXBS, 0);
	sblive_writesynth(sb_hw, FXBA, 0);
	sblive_writesynth(sb_hw, ADCBS, 0);
	sblive_writesynth(sb_hw, ADCBA, 0);

	/* disable channel interrupt */
	outl(0,  sb_hw->intrenabaddx);
	sblive_writesynth(sb_hw, CLIEL, 0);
	sblive_writesynth(sb_hw, CLIEH, 0);
	sblive_writesynth(sb_hw, SOLEL, 0);
	sblive_writesynth(sb_hw, SOLEH, 0);

	/* init envelope engine */
	for (nCh = 0; nCh < NUM_G; nCh++) {
		sblive_writesynth(sb_hw, nCh | DCYSUSV, ENV_OFF);
		sblive_writesynth(sb_hw, nCh | IP, 0);
		sblive_writesynth(sb_hw, nCh | VTFT, 0xFFFFL);
		sblive_writesynth(sb_hw, nCh | CVCF, 0xFFFFL);
		sblive_writesynth(sb_hw, nCh | PTRX, 0L);
		sblive_writesynth(sb_hw, nCh | CPF, 0L);
		sblive_writesynth(sb_hw, nCh | CCTRL, 0L);

		sblive_writesynth(sb_hw, nCh | PSST, 0L);
		sblive_writesynth(sb_hw, nCh | CSL, 0x10L);
		sblive_writesynth(sb_hw, nCh | CCCA, 0L);
		sblive_writesynth(sb_hw, nCh | Z1, 0L);
		sblive_writesynth(sb_hw, nCh | Z2, 0L);
		sblive_writesynth(sb_hw, nCh | FXRT, 0xD01C0000);

		sblive_writesynth(sb_hw, nCh | ATKHLD, 0);
		sblive_writesynth(sb_hw, nCh | DCYSUS, 0);
		sblive_writesynth(sb_hw, nCh | IFATN, 0xffff);
		sblive_writesynth(sb_hw, nCh | PEFE, 0);
		sblive_writesynth(sb_hw, nCh | FMMOD, 0);
		sblive_writesynth(sb_hw, nCh | TREMFRQ, 24);	/* 1 Hz */
		sblive_writesynth(sb_hw, nCh | FM2FRQ2, 24);	/* 1 Hz */
		sblive_writesynth(sb_hw, nCh | TEMPENV, 0);

		/*** these are last so OFF prevents writing ***/
		sblive_writesynth(sb_hw, nCh | LFOVAL2, 0);
		sblive_writesynth(sb_hw, nCh | LFOVAL1, 0);
		sblive_writesynth(sb_hw, nCh | ATKHLDV, 0);
		sblive_writesynth(sb_hw, nCh | ENVVOL, 0);
		sblive_writesynth(sb_hw, nCh | ENVVAL, 0);
	}


	/*
	   ** Init to 0x02109204 :
	   ** Clock accuracy    = 0     (1000ppm)
	   ** Sample Rate       = 2     (48kHz)
	   ** Audio Channel     = 1     (Left of 2)
	   ** Source Number     = 0     (Unspecified)
	   ** Generation Status = 1     (Original for Cat Code 12)
	   ** Cat Code          = 12    (Digital Signal Mixer)
	   ** Mode              = 0     (Mode 0)
	   ** Emphasis          = 0     (None)
	   ** CP                = 1     (Copyright unasserted)
	   ** AN                = 0     (Audio data)
	   ** P                 = 0     (Consumer)
	 */
	sblive_writesynth(sb_hw, SPCS0, 0x02109204);	// SPDIF0

	sblive_writesynth(sb_hw, SPCS1, 0x02109204);	// SPDIF1

	sblive_writesynth(sb_hw, SPCS2, 0x02109204);	// SPDIF2 & SPDIF3

	sblive_fxInit(sb_hw);	// initialize effects engine

	size = TMEMSIZE;
	sizeIdx = TMEMSIZEREG;
	while (sizeIdx) {
#ifdef TANKMEM
#if LINUX_VERSION_CODE < 0x020100
		sb_hw->tmemphysaddr =
		    __get_free_pages(GFP_KERNEL, sizeIdx + 2, 0);	// DMA flag for ISA bus only
#else
		sb_hw->tmemphysaddr =
		    __get_free_pages(GFP_KERNEL, sizeIdx + 2);
#endif

		if (!sb_hw->tmemphysaddr)
			status = CTSTATUS_NOMEMORY;
		else {		// Successfully allocated

			status = CTSTATUS_SUCCESS;
			sb_hw->tankmem = sb_hw->tmemphysaddr;
			sb_hw->tankmemptr = phys_to_virt(sb_hw->tmemphysaddr);
			sb_hw->tmemsizeIdx = sizeIdx;
		}
#else				// !TANKMEM
		sb_hw->tmemphysaddr = 0;
		sb_hw->tankmem = 0;
		sb_hw->tankmemptr = NULL;
		sb_hw->tmemsizeIdx = 0;
		size = 0;
		status = CTSTATUS_SUCCESS;
#endif				// TANKMEM
		if (status == CTSTATUS_SUCCESS)
			break;
		size /= 2;
		sizeIdx -= 1;
	}

	if (status == CTSTATUS_SUCCESS) {
		u16 count;

		sb_hw->tmemsize = size;

#if LINUX_VERSION_CODE < 0x020100
		sb_hw->PTBphysaddr = __get_free_pages(GFP_KERNEL, 4, 0);
#else
		sb_hw->PTBphysaddr = __get_free_pages(GFP_KERNEL, 4);
#endif
		if (!sb_hw->PTBphysaddr)
			status = CTSTATUS_ERROR;
		else {
#ifdef MEMTEST
			PDEBUG("\nhwaccess.c: __get_free_pages: [%lx]\n", sb_hw->PTBphysaddr);
#endif
			status = CTSTATUS_SUCCESS;
			sb_hw->virtualpagetable = sb_hw->PTBphysaddr;
			// In cardwo.c, we'll access through virtualpagetableptr to fill in
			// the page table, don't need to do phys_to_virt to convert
			// pVirtalPageTable to virtual address.
			sb_hw->virtualpagetableptr = (void *) sb_hw->PTBphysaddr;
		}

		if (status != CTSTATUS_SUCCESS) {
			ASSERT(FALSE);
			osFreeMemPhysical(sb_hw->tankmem);

			return status;
		}
		status = osAllocMemPhysical(EMUPAGESIZE,
					    &sb_hw->silentpage,
					    (void **)&sb_hw->silentpageptr,
					    &sb_hw->silentpagephysaddx);
		if (status != CTSTATUS_SUCCESS) {
			ASSERT(FALSE);
			osFreeMemPhysical(sb_hw->tankmem);
			osFreeMemPhysical(sb_hw->virtualpagetable);
			return status;
		} else
			memset(sb_hw->silentpageptr, 0, EMUPAGESIZE);

		/* init page table */
		for (count = 0; count < (MAXPAGES + 1024); count++) {
			/*
			   ** All the entries are inialized to point to itself
			   ** so that when illegal memory access occurs, the system
			   ** should not hang.
			 */
			sb_hw->virtualpagetableptr[count] =
			    (sb_hw->silentpagephysaddx * 2) | count;
		}

		/* init page table & tank memory base register */
		// get_free_pages returns virtual address, we need to convert it
		// to physical address and write it to page table base register.
		sblive_writesynth(sb_hw, PTB, virt_to_bus((void *) sb_hw->PTBphysaddr));
		sblive_writesynth(sb_hw, TMBA, sb_hw->tmemphysaddr);
		sblive_writesynth(sb_hw, TMBS, sizeIdx);
	} else {
		sb_hw->tmemsize = 0;
		return CTSTATUS_ERROR;
	}

	for (nCh = 0; nCh < NUM_G; nCh++) {
		sblive_writesynth(sb_hw, MAPA | nCh ,
				  0x1fff | (sb_hw->silentpagephysaddx * 2));
		sblive_writesynth(sb_hw, MAPB | nCh ,
				  0x1fff | (sb_hw->silentpagephysaddx * 2));
	}

	/*
	   ** Hokay, now enable the AUD bit
	   **  Enable Audio = 1
	   **  Mute Disable Audio = 0
	   **  Lock Tank Memory = 1
	   **  Lock Sound Memory = 0
	   **  Auto Mute = 1
	 */
	sblive_rmwac97(sb_hw, AC97_REG_MASTER_VOL, 0x8000, 0x8000);

	sblive_writeac97(sb_hw, AC97_REG_MASTER_VOL, 0);
	sblive_writeac97(sb_hw, AC97_REG_PCM_VOL, 0);

	if (sb_hw->dsCardCfg.chiprev < 6)
		outl(1+4+16, sb_hw->hwconfigaddx);
	else
		// With on-chip joystick
		outl(1+4+16+512, sb_hw->hwconfigaddx);


	// TOSLink detection
	sb_hw->has_toslink = 0;
	size = inl(sb_hw->hwconfigaddx);
	if (size & 0x6000) {
		volatile unsigned delay;
		outl(size|0x800, sb_hw->hwconfigaddx);
		for (delay = 0; delay < 512; delay++);	// small delay

		if (size != (inl(sb_hw->hwconfigaddx) & ~0x800u)) {
			sb_hw->has_toslink = 1;
			outl(size, sb_hw->hwconfigaddx);
		}
	}
	return CTSTATUS_SUCCESS;
}



/** channel status message lengths **/
u8 gabMsgLenChannel[] =
{
	3,			/* 0x80 note off        */
	3,			/* 0x90 note on         */
	3,			/* 0xA0 key pressure    */
	3,			/* 0xB0 control change  */
	2,			/* 0xC0 program change  */
	2,			/* 0xD0 channel pressure */
	3,			/* 0xE0 pitch bend      */
	1
};

/** system status message lengths **/
u8 gabMsgLenSystem[] =
{
	1,			/* 0xF0 sysex begin     */
	2,			/* 0xF1 midi tcqf       */
	3,			/* 0xF2 song position   */
	2,			/* 0xF3 song select     */
	1,			/* 0xF4 undefined       */
	1,			/* 0xF5 undefined       */
	1,			/* 0xF6 tune request    */
	1,			/* 0xF7 sysex eox       */

	1,			/* 0xF8 timing clock    */
	1,			/* 0xF9 undefined       */
	1,			/* 0xFA start           */
	1,			/* 0xFB continue        */
	1,			/* 0xFC stop            */
	1,			/* 0xFD undefined       */
	1,			/* 0xFE active sensing  */
	1			/* 0xFF system reset    */
};


/****************************************************************************/
/** Function : srToPitch                                                   **/
/**                                                                        **/
/** Input    : sampleRate - sampling rate                                  **/
/**                                                                        **/
/** Return   : pitch value                                                 **/
/**                                                                        **/
/** About    : convert sampling rate to pitch                              **/
/**                                                                        **/
/** Note     : for 8010, sampling rate is at 48kHz, this function should   **/
/**            be changed.                                                 **/
/****************************************************************************/
u32
srToPitch(u32 sampleRate)
{
	int i;
	static u32 logMagTable[128] = {
		0x00000, 0x02dfc, 0x05b9e, 0x088e6, 0x0b5d6, 0x0e26f, 0x10eb3, 0x13aa2,
		0x1663f, 0x1918a, 0x1bc84, 0x1e72e, 0x2118b, 0x23b9a, 0x2655d, 0x28ed5,
		0x2b803, 0x2e0e8, 0x30985, 0x331db, 0x359eb, 0x381b6, 0x3a93d, 0x3d081,
		0x3f782, 0x41e42, 0x444c1, 0x46b01, 0x49101, 0x4b6c4, 0x4dc49, 0x50191,
		0x5269e, 0x54b6f, 0x57006, 0x59463, 0x5b888, 0x5dc74, 0x60029, 0x623a7,
		0x646ee, 0x66a00, 0x68cdd, 0x6af86, 0x6d1fa, 0x6f43c, 0x7164b, 0x73829,
		0x759d4, 0x77b4f, 0x79c9a, 0x7bdb5, 0x7dea1, 0x7ff5e, 0x81fed, 0x8404e,
		0x86082, 0x88089, 0x8a064, 0x8c014, 0x8df98, 0x8fef1, 0x91e20, 0x93d26,
		0x95c01, 0x97ab4, 0x9993e, 0x9b79f, 0x9d5d9, 0x9f3ec, 0xa11d8, 0xa2f9d,
		0xa4d3c, 0xa6ab5, 0xa8808, 0xaa537, 0xac241, 0xadf26, 0xafbe7, 0xb1885,
		0xb3500, 0xb5157, 0xb6d8c, 0xb899f, 0xba58f, 0xbc15e, 0xbdd0c, 0xbf899,
		0xc1404, 0xc2f50, 0xc4a7b, 0xc6587, 0xc8073, 0xc9b3f, 0xcb5ed, 0xcd07c,
		0xceaec, 0xd053f, 0xd1f73, 0xd398a, 0xd5384, 0xd6d60, 0xd8720, 0xda0c3,
		0xdba4a, 0xdd3b4, 0xded03, 0xe0636, 0xe1f4e, 0xe384a, 0xe512c, 0xe69f3,
		0xe829f, 0xe9b31, 0xeb3a9, 0xecc08, 0xee44c, 0xefc78, 0xf148a, 0xf2c83,
		0xf4463, 0xf5c2a, 0xf73da, 0xf8b71, 0xfa2f0, 0xfba57, 0xfd1a7, 0xfe8df
	};

	static char logSlopeTable[128] = {
		0x5c, 0x5c, 0x5b, 0x5a, 0x5a, 0x59, 0x58, 0x58,
		0x57, 0x56, 0x56, 0x55, 0x55, 0x54, 0x53, 0x53,
		0x52, 0x52, 0x51, 0x51, 0x50, 0x50, 0x4f, 0x4f,
		0x4e, 0x4d, 0x4d, 0x4d, 0x4c, 0x4c, 0x4b, 0x4b,
		0x4a, 0x4a, 0x49, 0x49, 0x48, 0x48, 0x47, 0x47,
		0x47, 0x46, 0x46, 0x45, 0x45, 0x45, 0x44, 0x44,
		0x43, 0x43, 0x43, 0x42, 0x42, 0x42, 0x41, 0x41,
		0x41, 0x40, 0x40, 0x40, 0x3f, 0x3f, 0x3f, 0x3e,
		0x3e, 0x3e, 0x3d, 0x3d, 0x3d, 0x3c, 0x3c, 0x3c,
		0x3b, 0x3b, 0x3b, 0x3b, 0x3a, 0x3a, 0x3a, 0x39,
		0x39, 0x39, 0x39, 0x38, 0x38, 0x38, 0x38, 0x37,
		0x37, 0x37, 0x37, 0x36, 0x36, 0x36, 0x36, 0x35,
		0x35, 0x35, 0x35, 0x34, 0x34, 0x34, 0x34, 0x34,
		0x33, 0x33, 0x33, 0x33, 0x32, 0x32, 0x32, 0x32,
		0x32, 0x31, 0x31, 0x31, 0x31, 0x31, 0x30, 0x30,
		0x30, 0x30, 0x30, 0x2f, 0x2f, 0x2f, 0x2f, 0x2f
	};

	if (sampleRate == 0)
		return (0);	/* Bail out if no leading "1" */
	sampleRate *= 11185;	/* Scale 48000 to 0x20002380 */
	for (i = 31; i > 0; i--) {
		if (sampleRate & 0x80000000) {	/* Detect leading "1" */
			return (u32) (((s32) (i - 15) << 20) +
			       logMagTable[0x7f & (sampleRate >> 24)] +
				      (0x7f & (sampleRate >> 17)) *
			     logSlopeTable[0x7f & (sampleRate >> 24)]);
		}
		sampleRate = sampleRate << 1;
	}

	return 0;		/* Should never reach this point */
}


/****************************************************************************/
/** Function : sumVolumeToAttenuation                                      **/
/**                                                                        **/
/** Input    : inputValue - input volume                                   **/
/**                                                                        **/
/** Return   : attenuation value                                           **/
/**                                                                        **/
/** About    : convert volume to attenuation                               **/
/****************************************************************************/
/* Returns an attenuation based upon a cumulative volume value */
/* Algorithm calculates 0x200 - 0x10 log2 (input) */
u8 sumVolumeToAttenuation(u32 value)
{
	u16 count = 16;
	s16 ans;

	if (value == 0)
		return 0xFF;

	/* Find first SET bit. This is the integer part of the value */
	while ((value & 0x10000) == 0) {
		value <<= 1;
		count--;
	}

	/* The REST of the data is the fractional part. */
	ans = (s16) (0x110 - ((count << 4) + ((value & 0x0FFFFL) >> 12)));
	if (ans > 0xFF)
		ans = 0xFF;

	return (u8) ans;
}


//*****************************************************************************
//*  sblive_readsynth ( struct sblive_hw * sb_hw, u32 reg )                   *
//*****************************************************************************

u32 sblive_readsynth(struct sblive_hw * sb_hw, u32 reg)
{
	u32 val;
	unsigned long flags;

	spin_lock_irqsave(&sb_hw->emu_lock, flags);
	outl(reg & 0x00ff003f, sb_hw->synthaddx);
	val = inl(sb_hw->synthaddx + 4);
	spin_unlock_irqrestore(&sb_hw->emu_lock, flags);

	if (reg & 0x1f001f00) {
		reg &= 0x1f001f00;
		reg >>= 8;
		val <<= 31 - (reg >> 16);
		val >>= 31 - (reg >> 16) + (reg & 31);
	}
	return val;
}


//*****************************************************************************
//*  sblive_writesynth ( struct sblive_hw * sb_hw, u32 reg, u32 ddata )       *
//*****************************************************************************

void sblive_writesynth(struct sblive_hw * sb_hw, u32 reg, u32 ddata)
{
	unsigned long flags;

	if (reg & 0x1f001f00) {
		u32 dmask = 0xFFFFFFFF;
		u32 dtemp = reg & 0x1f001f00;
		dtemp >>= 8;
		dmask <<= 31 - (dtemp >> 16);
		dmask >>= 31 - (dtemp >> 16) + (dtemp & 31);
		dmask = (dmask << (dtemp & 31));
		ddata = (ddata << (dtemp & 31)) & dmask;
		ddata |= sblive_readsynth(sb_hw, reg & ~0x1f001f00) & ~dmask;
	}
	spin_lock_irqsave(&sb_hw->emu_lock, flags);
	outl(reg & 0x00ff003f, sb_hw->synthaddx);
	outl(ddata,sb_hw->synthaddx + 4);
	spin_unlock_irqrestore(&sb_hw->emu_lock, flags);
}


//*****************************************************************************
//*  halEfxRead ( struct sblive_hw * sb_hw, u32 reg )                         *
//*****************************************************************************
u32 halEfxRead(struct sblive_hw * sb_hw, u32 reg)
{
	u32 val;
	unsigned long flags;

	spin_lock_irqsave(&sb_hw->emu_lock, flags);
	outl(reg<<16, sb_hw->synthaddx);
	val = inl(sb_hw->synthaddx + 4);
	spin_unlock_irqrestore(&sb_hw->emu_lock, flags);

	return val;
}


//*****************************************************************************
//*  halEfxWrite ( struct sblive_hw * sb_hw, u32 reg, u32 ddata )             *
//*****************************************************************************
void halEfxWrite(struct sblive_hw * sb_hw, u32 reg, u32 ddata)
{
	unsigned long flags;
	spin_lock_irqsave(&sb_hw->emu_lock, flags);
	outl(reg << 16, sb_hw->synthaddx);
	outl(ddata, sb_hw->synthaddx + 4);
	spin_unlock_irqrestore(&sb_hw->emu_lock, flags);
}


//*****************************************************************************
//*  halReadPTR ( struct sblive_hw * sb_hw )                                             *
//*****************************************************************************
u32 halReadPTR(struct sblive_hw * sb_hw)
{
	return inl(sb_hw->synthaddx);
}


//*****************************************************************************
//*  halWritePTR ( struct sblive_hw * sb_hw, u8 byPTR )                       *
//*****************************************************************************
void halWritePTR(struct sblive_hw * sb_hw, u32 ptr)
{
	outl(ptr, sb_hw->synthaddx);
}

//*****************************************************************************
//*  halGetTimerDelay ( struct sblive_hw * sb_hw )                             *
//*****************************************************************************
u32 halGetTimerDelay(struct sblive_hw * sb_hw)
{
	return inl(sb_hw->timeraddx) & 0x3ff;
}


//*****************************************************************************
//*  halSetTimerDelay ( struct sblive_hw * sb_hw, u32 delay )                   *
//*****************************************************************************
void halSetTimerDelay(struct sblive_hw * sb_hw, u32 delay)
{
	delay += inl(sb_hw->timeraddx) & ~0x3ff;
	outl(delay, sb_hw->timeraddx);
}

//*****************************************************************************
//*  halIntrStatus ( struct sblive_hw * sb_hw, u32 intr )                     *
//*****************************************************************************
u32 halIntrStatus(struct sblive_hw * sb_hw, u32 intr)
{
	u32 status;

	status = inl(sb_hw->intrStatusAddx);

	return (intr & status);
}


//*****************************************************************************
//*  halIntrAck ( struct sblive_hw * sb_hw, u32 intr )                        *
//*****************************************************************************
void halIntrAck(struct sblive_hw * sb_hw, u32 intr)
{
	outl(intr, sb_hw->intrStatusAddx);
}


//*****************************************************************************
//*  halIntrEnable ( struct sblive_hw * sb_hw, u32 intrenb )                  *
//*****************************************************************************
void halIntrEnable(struct sblive_hw * sb_hw, u32 intrenb)
{
	u32 enable;
	unsigned long flags;

	spin_lock_irqsave(&sb_hw->emu_lock, flags);
	enable = inl(sb_hw->intrenabaddx);
	enable |= intrenb;
	outl(enable, sb_hw->intrenabaddx);
	spin_unlock_irqrestore(&sb_hw->emu_lock, flags);
}


//*****************************************************************************
//*  halIntrDisable ( struct sblive_hw * sb_hw, u32 intrenb )                 *
//*****************************************************************************
void halIntrDisable(struct sblive_hw * sb_hw, u32 intrenb)
{
	u32 enable;
	unsigned long flags;

	spin_lock_irqsave(&sb_hw->emu_lock, flags);
	enable = inl(sb_hw->intrenabaddx);
	enable &= ~intrenb;
	outl(enable, sb_hw->intrenabaddx);
	spin_unlock_irqrestore(&sb_hw->emu_lock, flags);
}


//*****************************************************************************
//*  halVoiceIntrEnable ( struct sblive_hw * sb_hw, u32 voicenum )            *
//*****************************************************************************
void halVoiceIntrEnable(struct sblive_hw * sb_hw, u32 voicenum)
{
	u32 CLIEval;

	/* voice interrupt */
	if (voicenum >= 32) {
		CLIEval = sblive_readsynth(sb_hw, CLIEH);
		CLIEval |= 1 << (voicenum - 32);
		sblive_writesynth(sb_hw, CLIEH, CLIEval);
	} else {
		CLIEval = sblive_readsynth(sb_hw, CLIEL);
		CLIEval |= 1 << voicenum;
		sblive_writesynth(sb_hw, CLIEL, CLIEval);
	}
}


//*****************************************************************************
//*  halSetStopOnLoop ( struct sblive_hw * sb_hw, u32 voicenum )              *
//*****************************************************************************
void halSetStopOnLoop(struct sblive_hw * sb_hw, u32 voicenum)
{
	u32 SOL;
	unsigned long flags;

	spin_lock_irqsave(&sb_hw->emu_lock, flags);

	/* voice interrupt */
	if (voicenum >= 32) {
		outl(SOLEH, sb_hw->synthaddx);
		SOL = inl(sb_hw->synthaddx + 4);
		SOL |= 1 << (voicenum - 32);
	} else {
		outl(SOLEL, sb_hw->synthaddx);
		SOL = inl(sb_hw->synthaddx + 4);
		SOL |= 1 << voicenum;
	}
	outl(SOL, sb_hw->synthaddx + 4);

	spin_unlock_irqrestore(&sb_hw->emu_lock, flags);
}


//*****************************************************************************
//*  halClearStopOnLoop ( struct sblive_hw * sb_hw, u32 voicenum )            *
//*****************************************************************************
void halClearStopOnLoop(struct sblive_hw * sb_hw, u32 voicenum)
{
	u32 SOL;
	unsigned long flags;

	spin_lock_irqsave(&sb_hw->emu_lock, flags);

	/* voice interrupt */
	if (voicenum >= 32) {
		outl(SOLEH, sb_hw->synthaddx);
		SOL = inl(sb_hw->synthaddx + 4);
		SOL &= ~(1 << (voicenum - 32));
	} else {
		outl(SOLEL, sb_hw->synthaddx);
		SOL = inl(sb_hw->synthaddx + 4);
		SOL &= ~(1 << voicenum);
	}
	outl(SOL, sb_hw->synthaddx + 4);

	spin_unlock_irqrestore(&sb_hw->emu_lock, flags);
}


//*****************************************************************************
//*  halVoiceIntrDisable ( struct sblive_hw * sb_hw, u32 voicenum )           *
//*****************************************************************************
void halVoiceIntrDisable(struct sblive_hw * sb_hw, u32 voicenum)
{
	u32 CLIEval;
	unsigned long flags;

	spin_lock_irqsave(&sb_hw->emu_lock, flags);

	/* voice interrupt */
	if (voicenum >= 32) {
		outl(CLIEH, sb_hw->synthaddx);
		CLIEval = inl(sb_hw->synthaddx + 4);
		CLIEval &= ~(1 << (voicenum - 32));
	} else {
		outl(CLIEL, sb_hw->synthaddx);
		CLIEval = inl(sb_hw->synthaddx + 4);
		CLIEval &= ~(1 << voicenum);
	}
	outl(CLIEval, sb_hw->synthaddx + 4);

	spin_unlock_irqrestore(&sb_hw->emu_lock, flags);
}


//*****************************************************************************
//*  halSetTOSLink ( struct sblive_hw * sb_hw, u32 setclear )                 *
//*****************************************************************************
int halSetTOSLink(struct sblive_hw * sb_hw, u32 setclear)
{
	u32 orig;

	if (!sb_hw->has_toslink)
		return CTSTATUS_NOTSUPPORTED;

	if (sb_hw->has_toslink == 1) {
		orig = inl(sb_hw->hwconfigaddx) & ~0x800u;
		if (setclear)
			orig |= 0x800;
		outl(orig, sb_hw->hwconfigaddx);
	} else {
		sblive_writesynth(sb_hw, SPBYPASS, setclear & 1);
	}

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//*  halGetTOSLink ( struct sblive_hw * sb_hw, u32 * setclear )               *
//*****************************************************************************
int halGetTOSLink(struct sblive_hw * sb_hw, u32 * setclear)
{
	u32 orig;

	if (!sb_hw->has_toslink)
		return CTSTATUS_NOTSUPPORTED;

	if (sb_hw->has_toslink == 1)
		orig = inl(sb_hw->hwconfigaddx) & 0x800u;
	else
		orig = sblive_readsynth(sb_hw, SPBYPASS) & 1;

	if (orig)
		*setclear = 1;
	else
		*setclear = 0;

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//*  halWC_WAIT ( struct sblive_hw * sb_hw, u32 wait )                        *
//*****************************************************************************
void halWC_WAIT(struct sblive_hw * sb_hw, u32 wait)
{
	volatile unsigned uCount;
	u32 newtime = 0, curtime;

	curtime = inl(sb_hw->wallclockaddx) >> 6;
	while (wait--) {
		uCount = 0;
		while (uCount++ < TIMEOUT) {
			newtime = inl(sb_hw->wallclockaddx) >> 6;
			if (newtime != curtime)
				break;
		}
		if (uCount >= TIMEOUT)
			break;
		curtime = newtime;
	}
}


//*****************************************************************************
//*  sblive_readac97 (struct sblive_hw * sb_hw, u8 index, u16 * pdata)        *
//*****************************************************************************
int sblive_readac97(struct sblive_hw * sb_hw, u8 index, u16 * pdata)
{
	unsigned long flags;
	spin_lock_irqsave(&sb_hw->emu_lock, flags);
	outb(index, sb_hw->mixeraddx + 2);
	*pdata = inw(sb_hw->mixeraddx);
	spin_unlock_irqrestore(&sb_hw->emu_lock, flags);

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//*  sblive_writeac97(struct sblive_hw * sb_hw, u8 index, u16 data)           *
//*****************************************************************************
int sblive_writeac97(struct sblive_hw * sb_hw, u8 index, u16 data)
{
	unsigned long flags;
	spin_lock_irqsave(&sb_hw->emu_lock, flags);
	outb(index, sb_hw->mixeraddx + 2);
	outw(data, sb_hw->mixeraddx);
	spin_unlock_irqrestore(&sb_hw->emu_lock, flags);

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//*  sblive_rmwac97(struct sblive_hw * sb_hw, u8 index, u16 data, u16 mask)   *
//*****************************************************************************
int sblive_rmwac97(struct sblive_hw * sb_hw, u8 index, u16 data, u16 mask)
{
	u16 temp;
	unsigned long flags;

	spin_lock_irqsave(&sb_hw->emu_lock, flags);
	outb(index, sb_hw->mixeraddx + 2);
	temp = inw(sb_hw->mixeraddx);
	temp &= ~mask;
	data &= mask;
	temp |= data;
	outw(temp, sb_hw->mixeraddx);
	spin_unlock_irqrestore(&sb_hw->emu_lock, flags);

	return CTSTATUS_SUCCESS;
}

/****************************************************************
            MPU access function
 ***************************************************************/

//*****************************************************************************
//* Function Name : hwmpuWriteData                                            *
//*                                                                           *
//* Description   : Writing data to MPU with timeout                          *
//*                                                                           *
//* Input         : sb_hw | Port object                                       *
//*                 bData | contain the command                               *
//*                                                                           *
//* Return        : int | contain the status of this call                     *
//*                                                                           *
//*****************************************************************************
int hwmpuWriteData(struct sblive_hw * sb_hw, u8 data)
{
	if ((inb(sb_hw->mpuaddx + 1) & 0x40) == 0) {
		outb(data,  sb_hw->mpuaddx);
		return CTSTATUS_SUCCESS;
	} else {
		return CTSTATUS_BUSY;
	}
}


//*****************************************************************************
//* Function Name : hwmpuReadData                                             *
//*                                                                           *
//* Description  : Reading data from MPU with timeout                         *
//*                                                                           *
//* Input         : sb_hw | Port object                                       *
//*                 bData | contain the command                               *
//*                                                                           *
//* Return        : int | contain the status of this call                     *
//*                                                                           *
//*****************************************************************************
int hwmpuReadData(struct sblive_hw * sb_hw, u8 * data)
{
	if ((inb(sb_hw->mpuaddx + 1) & 0x80) == 0) {
		*data = inb(sb_hw->mpuaddx);
		return CTSTATUS_SUCCESS;
	}
	return CTSTATUS_NODATA;
}


//*****************************************************************************
//* Function Name : hwmpuReset                                                *
//*                                                                           *
//* Description  : Hardware reset the MPU                                     *
//*                                                                           *
//* Input         : sb_hw | Port object                                       *
//*                                                                           *
//* Return        : int | contain the status of this call                     *
//*                                                                           *
//*****************************************************************************
int hwmpuReset(struct sblive_hw * sb_hw)
{
	u8 status;

	if (sb_hw->mpuacqcount == 0) {
		outb(0xff, sb_hw->mpuaddx + 1);
		halWC_WAIT(sb_hw, 8);
		outb(0xff, sb_hw->mpuaddx + 1);
		halWC_WAIT(sb_hw, 8);
		outb(0x3f, sb_hw->mpuaddx + 1);
		halWC_WAIT(sb_hw, 8);
		status = inb(sb_hw->mpuaddx);
		if (status == 0xFE)
			return CTSTATUS_SUCCESS;
		else
			return CTSTATUS_ERROR;
	}
	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//* Function Name : hwmpuAcquire                                              *
//*                                                                           *
//* Description   : Increase MPU acquire count                                *
//*                                                                           *
//* Input         : sb_hw | Port object                                       *
//*                                                                           *
//* Return        : int | contain the status of this call                     *
//*                                                                           *
//*****************************************************************************
int hwmpuAcquire(struct sblive_hw * sb_hw)
{
	++sb_hw->mpuacqcount;

	return CTSTATUS_SUCCESS;
}


//*****************************************************************************
//* Function Name : hwmpuRelease                                              *
//*                                                                           *
//* Description   : Decrease MPU acquire count                                *
//*                                                                           *
//* Input         : sb_hw | Port object                                       *
//*                                                                           *
//* Return        : int | contain the status of this call                     *
//*                                                                           *
//*****************************************************************************
int hwmpuRelease(struct sblive_hw * sb_hw)
{
	--sb_hw->mpuacqcount;

	return CTSTATUS_SUCCESS;
}


/****************************************************************
 *  Function Name : hwMcuReset
 *
 *  Description  : Reset the legacy SB.
 *
 *  Input        : wSbBase | SB I/O base address
 *
 *  Return        : int | contain the status of this call
 *
 ***************************************************************/
int hwMcuReset(u16 wSbBase)
{
	u32 i;

	/* reset DSP, read back 0xAA */
	outb(1, (u16) (wSbBase + SB_PORT_RESET));
	mdelay(10);
	outb(0, (u16) (wSbBase + SB_PORT_RESET));

	for (i = 0; i < 65536; i++) {
		if (inb((u16) (wSbBase + SB_PORT_READSTATUS)) & 0x80) {
			if (inb((u16) (wSbBase + SB_PORT_READDATA)) == 0xAA)
				return CTSTATUS_SUCCESS;
		}
	}

	return CTSTATUS_ERROR;
}


/****************************************************************
 *  Function Name : hwMcuGetIrqStatus
 *
 *  Description  : Retrieve the SB Irq status.
 *
 *  Input        : wSbBase | SB I/O base address
 *
 *  Return        : int | contain the status of this call
 *
 ***************************************************************/
int hwMcuGetIrqStatus(u16 wSbBase, u8 * pstatus)
{
	outb(MIXER_REG_IRQSTATUS, (u16) (wSbBase + SB_PORT_MIXERINDEX));
	*pstatus = inb((u16) (wSbBase + SB_PORT_MIXERDATA));

	return CTSTATUS_SUCCESS;
}
