/*     
 **********************************************************************
 *     sblive_irqmgr.c - IRQ manager for emu10k1 driver 
 *     Copyright 1999, 2000 Creative Labs, Inc. 
 * 
 ********************************************************************** 
 * 
 *     Date                 Author          Summary of changes 
 *     ----                 ------          ------------------ 
 *     October 20, 1999     Bertrand Lee    base code release 
 * 
 ********************************************************************** 
 * 
 *     This program is free software; you can redistribute it and/or 
 *     modify it under the terms of the GNU General Public License as 
 *     published by the Free Software Foundation; either version 2 of 
 *     the License, or (at your option) any later version. 
 * 
 *     This program is distributed in the hope that it will be useful, 
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *     GNU General Public License for more details. 
 * 
 *     You should have received a copy of the GNU General Public 
 *     License along with this program; if not, write to the Free 
 *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, 
 *     USA. 
 * 
 ********************************************************************** 
 */

#include "hwaccess.h"

/************************************************************************
*
*   EMU IRQ Manager
*
************************************************************************/

/************************************************************************
*
*   int    sblive_irqmgrCallback(unsigned long event,
*                              unsigned long refdata,
*                              unsigned long param)
*
*   ENTRY
*       event -   Reason for callback (currently not used)
*       refdata
*               -   card object
*       param -   Parameter (currently not used)
*
*   RETURNS
*       Always returns CTSTATUS_SUCCESS
*
*   ABOUT
*       This function is called by the VxD's IRQ handler and
*       will dispatch the irq to the appropriate handler. If
*       no handler is present for a service, the interrupt
*       for that service is disabled.
*
************************************************************************/
int sblive_irqmgrCallback(unsigned long event, unsigned long refdata, unsigned long param)
{
	u32 irqstatus;
	u32 nomask = 0xffffffff;
	struct sblive_hw *sb_hw = (struct sblive_hw *) refdata;
	u32 ptr;
	u32 serviced;


	TESTMAGIC(sb_hw);

	// preserve PTR register
	ptr = halReadPTR(sb_hw);

	/*
	   ** NOTE :
	   ** We do a 'while loop' here cos on certain machines, with both
	   ** playback and recording going on at the same time, IRQs will
	   ** stop coming in after a while. Checking IPND indeed shows that
	   ** there are interrupts pending but the PIC says no IRQs pending.
	   ** I suspect that some boards need edge-triggered IRQs but are not
	   ** getting that condition if we don't completely clear the IPND
	   ** (make sure no more interrupts are pending).
	   ** - Eric
	 */

	irqstatus = halIntrStatus(sb_hw, nomask);
	serviced = irqstatus;

	do {
		//        DPD("sblive_irqmgrCallback ", irqstatus);

		if (irqstatus & IRQTYPE_PCIBUSERROR) {
			if (sb_hw->PCIBusErrorCallback != NULL)
				sb_hw->PCIBusErrorCallback(0, sb_hw->pcierrorrefdata,
					irqstatus & IRQTYPE_PCIBUSERROR);
			else
				sblive_irqmgrDisableIrq(sb_hw, ENB_PCI);
		}
		if (irqstatus & IRQTYPE_MIXERBUTTON) {
			if (sb_hw->MixerButtonCallback != NULL)
				sb_hw->MixerButtonCallback(0, sb_hw->mixerbuttonrefdata,
					irqstatus & IRQTYPE_MIXERBUTTON);
			else
				sblive_irqmgrDisableIrq(sb_hw, ENB_VINC | ENB_VDEC | ENB_MUTE);
		}
		if (irqstatus & IRQTYPE_VOICE) {
			if (sb_hw->VoiceCallback != NULL)
				sb_hw->VoiceCallback(0, sb_hw->voicerefdata,
					      irqstatus & IRQTYPE_VOICE);
		}
		if (irqstatus & IRQTYPE_RECORD) {
			if (sb_hw->RecordCallback != NULL)
				sb_hw->RecordCallback(0, sb_hw->recordrefdata,
					     irqstatus & IRQTYPE_RECORD);
			else
				sblive_irqmgrDisableIrq(sb_hw, ENB_ADC);
		}
		if (irqstatus & IRQTYPE_MPUOUT) {
			if (sb_hw->MpuOutCallback != NULL)
				sb_hw->MpuOutCallback(0, sb_hw->mpuoutrefdata,
					     irqstatus & IRQTYPE_MPUOUT);
			else
				sblive_irqmgrDisableIrq(sb_hw, ENB_TX);
		}
		if (irqstatus & IRQTYPE_MPUIN) {
			if (sb_hw->MpuInCallback != NULL)
				sb_hw->MpuInCallback(0, sb_hw->mpuinrefdata,
					      irqstatus & IRQTYPE_MPUIN);
			else
				sblive_irqmgrDisableIrq(sb_hw, ENB_RX);
		}
		if (irqstatus & IRQTYPE_TIMER) {
			if (sb_hw->TimerCallback != NULL)
				sb_hw->TimerCallback(0, sb_hw->timerrefdata,
					      irqstatus & IRQTYPE_TIMER);
			else
				sblive_irqmgrDisableIrq(sb_hw, ENB_TIMER);
		}
		if (irqstatus & IRQTYPE_SPDIF) {
			if (sb_hw->SPDIFCallback != NULL)
				sb_hw->SPDIFCallback(0, sb_hw->spdifrefdata,
					      irqstatus & IRQTYPE_SPDIF);
			else
				sblive_irqmgrDisableIrq(sb_hw, ENB_GPSCS | ENB_CDCS);
		}
		if (irqstatus & IRQTYPE_DSP) {
			if (sb_hw->DSPCallback != NULL)
				sb_hw->DSPCallback(0, sb_hw->DSPrefdata,
						irqstatus & IRQTYPE_DSP);
			else
				sblive_irqmgrDisableIrq(sb_hw, ENB_DSP);
		}

		halIntrAck(sb_hw, irqstatus);

	} while ((irqstatus = halIntrStatus(sb_hw, nomask)));

	if (serviced)
		irqAcknowledge(&sb_hw->hw_irq);


	// restore PTR register
	halWritePTR(sb_hw, ptr);


	if (serviced)
		return CTSTATUS_SUCCESS;
	else
		return CTSTATUS_ERROR;
}


/************************************************************************
*
*   int    sblive_irqmgrInit(struct sblive_hw * sb_hw)
*
*   ENTRY
*       sb_hw -   Card object
*
*   RETURNS
*       SUCCESS -   CTSTATUS_SUCCESS
*       FAILURE -   CTSTATUS_ERROR
*
*   ABOUT
*       Initialization
*
************************************************************************/
int sblive_irqmgrInit(struct sblive_hw *sb_hw)
{
	TESTMAGIC(sb_hw);



	if (irqInit(&sb_hw->hw_irq, sb_hw->irq,
		    sblive_irqmgrCallback, (u32) sb_hw) != CTSTATUS_SUCCESS) {
		DPF("CARD : Error initialising Irq");
		return CTSTATUS_ERROR;
	} else {
		if (irqAcquire(&sb_hw->hw_irq) == CTSTATUS_SUCCESS)
			irqStart(&sb_hw->hw_irq);
		else {
			DPF("CARD : Irq Acquire Fail");
			return CTSTATUS_ERROR;
		}
	}


	return CTSTATUS_SUCCESS;
}


/************************************************************************
*
*   int    sblive_irqmgrExit(struct sblive_hw * sb_hw)
*
*   ENTRY
*       sb_hw -   Card object
*
*   RETURNS
*       Always returns CTSTATUS_SUCCESS
*
*   ABOUT
*       Shutdown code
*
************************************************************************/
int sblive_irqmgrExit(struct sblive_hw *sb_hw)
{
	// TODO : disable all card irqs

	irqRelease(&sb_hw->hw_irq);
	irqStop(&sb_hw->hw_irq);
	irqExit(&sb_hw->hw_irq);

	return CTSTATUS_SUCCESS;
}


/************************************************************************
*
*   int    sblive_irqmgrInstallIrqHandler(struct sblive_hw * sb_hw,
*                                       u32 irqtype,
*                                       CALLBACKFN callback,
*                                       u32 refdata)
*
*   ENTRY
*       sb_hw -   Card object
*       wIrqType-   type of service to install handler for
*       callback
*               -   irq service handler
*       refdata
*               -   data to call hadler with
*
*   RETURNS
*       SUCCESS -   CTSTATUS_SUCCESS
*       FAILURE -   CTSTATUS_ERROR
*
*   ABOUT
*       Used to install handlers for various irq services
*
************************************************************************/
int sblive_irqmgrInstallIrqHandler(
				   struct sblive_hw *sb_hw,
				   u32 irqtype,
				   CALLBACKFN callback,
				   u32 refdata
)
{
	TESTMAGIC(sb_hw);
	ASSERT(callback);

	DPD("sblive_irqmgrInstallIrqHandler ", irqtype);

	switch (irqtype) {
	case IRQTYPE_PCIBUSERROR:
		if (sb_hw->PCIBusErrorCallback != NULL)
			return CTSTATUS_INUSE;
		sb_hw->PCIBusErrorCallback = callback;
		sb_hw->pcierrorrefdata = refdata;
		break;

	case IRQTYPE_MIXERBUTTON:
		if (sb_hw->MixerButtonCallback != NULL)
			return CTSTATUS_INUSE;
		sb_hw->MixerButtonCallback = callback;
		sb_hw->mixerbuttonrefdata = refdata;
		break;

	case IRQTYPE_VOICE:
		if (sb_hw->VoiceCallback != NULL)
			return CTSTATUS_INUSE;
		sb_hw->VoiceCallback = callback;
		sb_hw->voicerefdata = refdata;
		break;

	case IRQTYPE_RECORD:
		if (sb_hw->RecordCallback != NULL)
			return CTSTATUS_INUSE;
		sb_hw->RecordCallback = callback;
		sb_hw->recordrefdata = refdata;
		break;

	case IRQTYPE_MPUOUT:
		if (sb_hw->MpuOutCallback != NULL)
			return CTSTATUS_INUSE;
		sb_hw->MpuOutCallback = callback;
		sb_hw->mpuoutrefdata = refdata;
		break;

	case IRQTYPE_MPUIN:
		if (sb_hw->MpuInCallback != NULL)
			return CTSTATUS_INUSE;
		sb_hw->MpuInCallback = callback;
		sb_hw->mpuinrefdata = refdata;
		break;

	case IRQTYPE_TIMER:
		if (sb_hw->TimerCallback != NULL)
			return CTSTATUS_INUSE;
		sb_hw->TimerCallback = callback;
		sb_hw->timerrefdata = refdata;
		break;

	case IRQTYPE_SPDIF:
		if (sb_hw->SPDIFCallback != NULL)
			return CTSTATUS_INUSE;
		sb_hw->SPDIFCallback = callback;
		sb_hw->spdifrefdata = refdata;
		break;

	case IRQTYPE_DSP:
		if (sb_hw->DSPCallback != NULL)
			return CTSTATUS_INUSE;
		sb_hw->DSPCallback = callback;
		sb_hw->DSPrefdata = refdata;
		break;

	default:
		return CTSTATUS_ERROR;
	}

	return CTSTATUS_SUCCESS;
}


/************************************************************************
*
*   int    sblive_irqmgrUninstallIrqHandler(struct sblive_hw * sb_hw,
*                                         u32 irqtype)
*
*   ENTRY
*       sb_hw -   Card object
*       wIrqType-   type of service to remove handler for
*
*   RETURNS
*       SUCCESS -   CTSTATUS_SUCCESS
*       FAILURE -   CTSTATUS_ERROR
*
*   ABOUT
*       Used to remove previously installed handlers
*
************************************************************************/

int sblive_irqmgrUninstallIrqHandler(struct sblive_hw *sb_hw, u32 irqtype)
{
	unsigned long flags;
	
	TESTMAGIC(sb_hw);
	DPD("sblive_irqmgrUninstallIrqHandler ", irqtype);

	spin_lock_irqsave(&sb_hw->emu_lock, flags);

	switch (irqtype) {
	case IRQTYPE_PCIBUSERROR:
		sb_hw->PCIBusErrorCallback = NULL;
		sb_hw->pcierrorrefdata = 0;
		break;

	case IRQTYPE_MIXERBUTTON:
		sb_hw->MixerButtonCallback = NULL;
		sb_hw->mixerbuttonrefdata = 0;
		break;

	case IRQTYPE_VOICE:
		sb_hw->VoiceCallback = NULL;
		sb_hw->voicerefdata = 0;
		break;

	case IRQTYPE_RECORD:
		sb_hw->RecordCallback = NULL;
		sb_hw->recordrefdata = 0;
		break;

	case IRQTYPE_MPUOUT:
		sb_hw->MpuOutCallback = NULL;
		sb_hw->mpuoutrefdata = 0;
		break;

	case IRQTYPE_MPUIN:
		sb_hw->MpuInCallback = NULL;
		sb_hw->mpuinrefdata = 0;
		break;

	case IRQTYPE_TIMER:
		sb_hw->TimerCallback = NULL;
		sb_hw->timerrefdata = 0;
		break;

	case IRQTYPE_SPDIF:
		sb_hw->SPDIFCallback = NULL;
		sb_hw->spdifrefdata = 0;
		break;

	case IRQTYPE_DSP:
		sb_hw->DSPCallback = NULL;
		sb_hw->DSPrefdata = 0;
		break;

	default:
		spin_unlock_irqrestore(&sb_hw->emu_lock, flags);
		return CTSTATUS_ERROR;
	}

	spin_unlock_irqrestore(&sb_hw->emu_lock, flags);

	return CTSTATUS_SUCCESS;
}


/************************************************************************
*
*   int    sblive_irqmgrEnableIrq(struct sblive_hw * sb_hw, u16 wIrqType)
*
*   ENTRY
*       sb_hw -   Card object
*       wIrqType-   type of service to enable interrupt for
*
*   RETURNS
*       Always returns CTSTATUS_SUCCESS
*
*   ABOUT
*       Enables the specified irq service
*
************************************************************************/

int sblive_irqmgrEnableIrq(struct sblive_hw *sb_hw, u32 irqtype)
{
	/*
	 * TODO :
	 * put protection here so that we don't accidentally
	 * screw-up another cardxxx objects irqs
	 */

	ASSERT(sb_hw);
	DPD("sblive_irqmgrEnableIrq ", irqtype);
	halIntrEnable(sb_hw, irqtype);

	return CTSTATUS_SUCCESS;
}


/************************************************************************
*
*   int    sblive_irqmgrDisableIrq(struct sblive_hw * sb_hw, u16 wIrqType)
*
*   ENTRY
*       sb_hw -   Card object
*       wIrqType-   type of service to disable interrupt for
*
*   RETURNS
*       Always returns CTSTATUS_SUCCESS
*
*   ABOUT
*       Disables the specified irq service
*
************************************************************************/

int sblive_irqmgrDisableIrq(struct sblive_hw *sb_hw, u32 irqtype)
{
	/*
	 * TODO :
	 * put protection here so that we don't accidentally
	 * screw-up another cardxxx objects irqs
	 */

	ASSERT(sb_hw);
	DPD("sblive_irqmgrDisableIrq ", irqtype);
	halIntrDisable(sb_hw, irqtype);

	return CTSTATUS_SUCCESS;
}


/************************************************************************
*
*   int    sblive_irqmgrEnableVoiceIrq(struct sblive_hw * sb_hw, u32 voicenum)
*
*   ENTRY
*       sb_hw -   Card object
*       voicenum
*               -   voice to enable interrupt for
*
*   RETURNS
*       Always returns CTSTATUS_SUCCESS
*
*   ABOUT
*       Enables the specified voice irq service
*
************************************************************************/

int sblive_irqmgrEnableVoiceIrq(struct sblive_hw *sb_hw, u32 voicenum)
{
	/*
	 * TODO :
	 * put protection here so that we don't accidentally
	 * screw-up another cardxxx objects irqs
	 */

	ASSERT(sb_hw);
	DPD("sblive_irqmgrEnableVoiceIrq ", voicenum);
	halVoiceIntrEnable(sb_hw, voicenum);

	return CTSTATUS_SUCCESS;
}


/************************************************************************
*
*   int    sblive_irqmgrDisableVoiceIrq(struct sblive_hw * sb_hw, u32 voicenum)
*
*   ENTRY
*       sb_hw -   Card object
*       voicenum
*               -   voice to disable interrupt for
*
*   RETURNS
*       Always returns CTSTATUS_SUCCESS
*
*   ABOUT
*       Disables the specified voice irq service
*
************************************************************************/

int sblive_irqmgrDisableVoiceIrq(struct sblive_hw *sb_hw, u32 voicenum)
{
	/*
	 * TODO :
	 * put protection here so that we don't accidentally
	 * screw-up another cardxxx objects irqs
	 */

	TESTMAGIC(sb_hw);

	DPD("sblive_irqmgrDisableVoiceIrq ", voicenum);
	halVoiceIntrDisable(sb_hw, voicenum);

	return CTSTATUS_SUCCESS;
}
