/*     
 **********************************************************************
 *     myirq.c - IRQ HAL for emu10k1 driver 
 *     Copyright 1999, 2000 Creative Labs, Inc. 
 * 
 ********************************************************************** 
 * 
 *     Date                 Author          Summary of changes 
 *     ----                 ------          ------------------ 
 *     October 20, 1999     Bertrand Lee    base code release 
 * 
 ********************************************************************** 
 * 
 *     This program is free software; you can redistribute it and/or 
 *     modify it under the terms of the GNU General Public License as 
 *     published by the Free Software Foundation; either version 2 of 
 *     the License, or (at your option) any later version. 
 * 
 *     This program is distributed in the hope that it will be useful, 
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *     GNU General Public License for more details. 
 * 
 *     You should have received a copy of the GNU General Public 
 *     License along with this program; if not, write to the Free 
 *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, 
 *     USA. 
 * 
 ********************************************************************** 
 */ 

#include "platform.h"
#include "irq.h"

int irqSetup(struct sblive_irq * irq_ptr, u8 irq, CALLBACKFN callback, u32 refdata);
int irqCleanup(struct sblive_irq * irq_ptr);
int irqMask(struct sblive_irq * irq_ptr);
int irqUnmask(struct sblive_irq * irq_ptr);
int irqEOI(struct sblive_irq * irq_ptr);

////////////////////////////////////////////////////////////////////////////////
int irqInit (struct sblive_irq * irq_ptr, u8 irq,
	     CALLBACKFN callback, u32 refdata)
{

    irq_ptr->status = 0;
    if (irqSetup(irq_ptr, irq, callback, refdata) == CTSTATUS_SUCCESS) {
        irq_ptr->status |= (FLAGS_ENABLED | FLAGS_AVAILABLE);
        irq_ptr->irq = irq;
    } else {
        irq_ptr->status &= ~FLAGS_ENABLED;
        return CTSTATUS_ERROR;
    }

    return CTSTATUS_SUCCESS;
}

////////////////////////////////////////////////////////////////////////////////
int irqExit (struct sblive_irq * irq_ptr)
{
    irqCleanup(irq_ptr);
    irq_ptr->status &= ~FLAGS_ENABLED;

    return CTSTATUS_SUCCESS;
}


////////////////////////////////////////////////////////////////////////////////
int irqAcquire (struct sblive_irq * irq_ptr)
{
    if ((irq_ptr->status & FLAGS_ENABLED) &&
        (irq_ptr->status & FLAGS_AVAILABLE))
        irq_ptr->status &= (~FLAGS_AVAILABLE);
    else
        return CTSTATUS_INUSE;

    DPD("IRQ : Acquired channel", irq_ptr->irq);
    return CTSTATUS_SUCCESS;
}


////////////////////////////////////////////////////////////////////////////////
int irqRelease (struct sblive_irq * irq_ptr)
{
    DPF("IRQ : Released");
    irq_ptr->status |= FLAGS_AVAILABLE;

    return CTSTATUS_SUCCESS;
}


////////////////////////////////////////////////////////////////////////////////
int irqStart(struct sblive_irq * irq_ptr)
{
    irqUnmask(irq_ptr);
    return CTSTATUS_SUCCESS;
}


////////////////////////////////////////////////////////////////////////////////
int irqStop(struct sblive_irq * irq_ptr)
{
    irqMask(irq_ptr);
    return CTSTATUS_SUCCESS;
}


////////////////////////////////////////////////////////////////////////////////
int irqAcknowledge(struct sblive_irq * irq_ptr)
{
    irqEOI(irq_ptr);

    return CTSTATUS_SUCCESS;
}

// Linux-specific stuff follows... //

#define EMU10K1_INTERRUPT_ID 0xfeebbeef

// Interrupt handler
void emu10k1_interrupt(int nIRQ, void *pvDevID, struct pt_regs *pRegs)
{
  struct sblive_irq * irq_ptr = pvDevID;

  if (irq_ptr->ID != EMU10K1_INTERRUPT_ID)
      return;

  DPD("emu10k1_interrupt called, nIRQ=", nIRQ);
  ASSERT(irq_ptr->callback);
  irq_ptr->callback(0, irq_ptr->refdata, 0);
}


////////////////////////////////////////////////////////////////////////////////
int irqSetup(struct sblive_irq * irq_ptr, u8 irq,
	     CALLBACKFN callback, u32 refdata)
{
    unsigned long flags;
    
    irq_ptr->ID = EMU10K1_INTERRUPT_ID;
    irq_ptr->callback  = callback;
    irq_ptr->refdata   = refdata;

#if LINUX_VERSION_CODE < 0x020100
    flags = SA_INTERRUPT | SA_SHIRQ;
#else
    flags = SA_SHIRQ;
#endif

    // Reserve IRQ Line
    if (request_irq(irq, emu10k1_interrupt, flags, "emu10k1", irq_ptr)) {
      printk(KERN_ERR "emu10k1: irq %u in use\n", (unsigned int) irq);
      return CTSTATUS_ERROR;
    } 

    return CTSTATUS_SUCCESS;
}


////////////////////////////////////////////////////////////////////////////////
int irqCleanup(struct sblive_irq * irq_ptr)
{

    irq_ptr->callback = NULL;
    irq_ptr->refdata = 0;

    free_irq(irq_ptr->irq, irq_ptr);

    return CTSTATUS_SUCCESS;
}


////////////////////////////////////////////////////////////////////////////////
int irqMask (struct sblive_irq * irq_ptr)
{
    return CTSTATUS_SUCCESS;
}


////////////////////////////////////////////////////////////////////////////////
int irqUnmask (struct sblive_irq * irq_ptr)
{
    return CTSTATUS_SUCCESS;
}

////////////////////////////////////////////////////////////////////////////////
int irqEOI (struct sblive_irq * irq_ptr)
{
    return CTSTATUS_SUCCESS;
}

