/*
 **********************************************************************
 *     recmgr.c -- Recording manager for emu10k1 driver
 *     Copyright 1999, 2000 Creative Labs, Inc.
 *
 **********************************************************************
 *
 *     Date                 Author          Summary of changes
 *     ----                 ------          ------------------
 *     October 20, 1999     Bertrand Lee    base code release
 *
 **********************************************************************
 *
 *     This program is free software; you can redistribute it and/or
 *     modify it under the terms of the GNU General Public License as
 *     published by the Free Software Foundation; either version 2 of
 *     the License, or (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public
 *     License along with this program; if not, write to the Free
 *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 *     USA.
 *
 **********************************************************************
 */

#include "hwaccess.h"
#include "recmgr.h"

/*************************************************************************** */
/**  Function: recmgrInit                                                  * */
/**                                                                        * */
/**  Input   : rec_ptr - pointer recording object                        * */
/**                                                                        * */
/**  About   : stop recording when init                                    * */
/*************************************************************************** */
int recmgrInit(struct record *rec_ptr)
{
	struct sblive_hw *hw_ptr = rec_ptr->sb_hw;

	/* disable record transfer */
	sblive_writesynth(hw_ptr, ADCBS, 0);
	sblive_writesynth(hw_ptr, ADCCR, 0);

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/**  Function: recmgrStartRecord                                           * */
/**                                                                        * */
/**  Input   : rec_ptr - pointer recording object                        * */
/**                                                                        * */
/**  About   : start recording                                             * */
/*************************************************************************** */
int recmgrStartRecord(struct record *rec_ptr)
{
	struct sblive_hw *hw_ptr = rec_ptr->sb_hw;
	u32 adcsr;

	/* set record host buffer address */
	/* make sure that physical address is page-aligned */
	ASSERT(!(rec_ptr->physaddx & 0xfff));

#ifdef RECTEST
	DPD(" data in 0x120 -> ", halEfxRead(hw_ptr, 0x120));
	DPD(" data in 0x121 -> ", halEfxRead(hw_ptr, 0x121));
#endif

	DPD("recmgrStartRecord: base addx:", rec_ptr->physaddx >> 12);
	sblive_writesynth(hw_ptr, ADCBA, rec_ptr->physaddx);

	switch (rec_ptr->samplingrate) {
	case 0xBB80:
		adcsr = ADC48K;
		break;
	case 0xAC44:
		adcsr = ADC44K;
		break;
	case 0x7D00:
		adcsr = ADC32K;
		break;
	case 0x5DC0:
		adcsr = ADC24K;
		break;
	case 0x5622:
		adcsr = ADC22K;
		break;
	case 0x3E80:
		adcsr = ADC16K;
		break;
	case 0x2B11:
		adcsr = ADC11K;
		break;
	case 0x1F40:
		adcsr = ADC8K;
		break;
	default:
		ASSERT(FALSE);
		return CTSTATUS_ERROR;
	}

	if (rec_ptr->is_stereo)
		sblive_writesynth(hw_ptr, ADCCR, adcsr | 0x18);
	else
		sblive_writesynth(hw_ptr, ADCCR, adcsr | 0x08);

	sblive_writesynth(hw_ptr, ADCBS, rec_ptr->bufsizereg);

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/**  Function: recmgrStopRecord                                            * */
/**                                                                        * */
/**  Input   : rec_ptr - pointer recording object                        * */
/**                                                                        * */
/**  About   : stop recording                                              * */
/*************************************************************************** */
int recmgrStopRecord(struct record *rec_ptr)
{
	struct sblive_hw *hw_ptr = rec_ptr->sb_hw;

	/* disable record transfer */
	sblive_writesynth(hw_ptr, ADCBS, 0);
	sblive_writesynth(hw_ptr, ADCCR, 0);

	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/**  Function: recmgrSetControl                                            * */
/**                                                                        * */
/**  Input   : rec_ptr - pointer recording object                        * */
/**            controlid - control ID                                    * */
/**            value - value to set for a specified control              * */
/**                                                                        * */
/**  About   : set recording resource                                      * */
/**                                                                        * */
/**  Note    :                                                             * */
/**                                                                        * */
/**  XD, DMA, WR and RIGHT bits in CCCA register controls the way to       * */
/**  record for 8005.                                                      * */
/**                                                                        * */
/**  XD   DMA    WR    RIGHT                                               * */
/**  1     1     0      0      --- record from external(AC97, SPDIF, MIC)  * */
/**  1     1     1      0      --- record from the output of Emu8000       * */
/**                                effects engine.                         * */
/**                                                                        * */
/*************************************************************************** */
int recmgrSetControl(struct record *rec_ptr, u32 controlid, u32 value)
{
	return CTSTATUS_SUCCESS;
}


/*************************************************************************** */
/**  Function: recmgrGetPos                                                * */
/**                                                                        * */
/**  Input   : rec_ptr - pointer recording object                        * */
/**            pos -  pointer to position returned                      * */
/**                                                                        * */
/**  About   : get recording position(no. of bytes per channel)            * */
/**                                                                        * */
/**  Formula : (WC*SamplingRate/48kHz)%(record buffer size)                * */
/*************************************************************************** */
int recmgrGetPos(struct record *rec_ptr, u32 *pos)
{
	struct sblive_hw *hw_ptr = rec_ptr->sb_hw;

	u32 curidx;

	curidx = recmgrGetRecIdx(hw_ptr);
	if (curidx >= rec_ptr->prevadcidx)
		*pos = curidx - rec_ptr->prevadcidx;
	else
		*pos = 0xfffff - (rec_ptr->prevadcidx - curidx);

	*pos = (*pos) * ((u16) rec_ptr->samplingrate) / 48000
		* 2 * (rec_ptr->is_stereo + 1);

	ASSERT(*pos <= rec_ptr->recbufsize);

	DPD(" Recmgr: *pos: ", *pos);
	if (rec_ptr->pong) {
		*pos += rec_ptr->recbufsize / 2;
		ASSERT(*pos <= rec_ptr->recbufsize);
	}

	return CTSTATUS_SUCCESS;
}



u32 recmgrGetRecIdx(struct sblive_hw *hw_ptr)
{
	return ((inl((u16) hw_ptr->wallclockaddx) >> 6) & 0xfffff);
	//return((u16)sblive_readsynth(hw_ptr, ADCIDX));
}
